#include "dailymotiongrouplistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "dailymotion.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

using namespace QtJson;

DailymotionGroupListModel::DailymotionGroupListModel(QObject *parent) :
    AbstractGroupListModel(parent),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionGroupListModel::clear() {
    AbstractGroupListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionGroupListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreGroups();
    }
}

void DailymotionGroupListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void DailymotionGroupListModel::getGroups(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }

    feed.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);

    this->setLoading(true);
    QNetworkReply *reply = Dailymotion::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void DailymotionGroupListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = Dailymotion::instance()->createSearchReply(Queries::Groups, this->query(), this->offset(), this->order());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void DailymotionGroupListModel::addGroups() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        this->connect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        Dailymotion::instance()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                GroupItem *group = new GroupItem;
                group->loadDailymotionGroup(entries.at(i).toMap());
                QMetaObject::invokeMethod(this, "appendGroup", Qt::QueuedConnection, Q_ARG(QSharedPointer<GroupItem>, QSharedPointer<GroupItem>(group)));
            }

            QTimer::singleShot(1000, this, SLOT(stopLoading()));
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        this->disconnect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionGroupListModel::getMoreGroups() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getGroups();
        }
    }
}
