#include "events.h"
#include "../../cutetubeapp/src/base/json.h"
#include "youtube.h"
#include "dailymotion.h"
#include "vimeo.h"
#include "database.h"
#include <QUrl>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QDBusInterface>
#include <QDomDocument>
#include <QDomElement>
#include <QStringList>
#include <QCoreApplication>
#include <QTimer>
#include <QFileInfo>
#include <QFile>
#include <QDir>
#include <QDateTime>
#include <QSettings>
#ifdef MEEGO_EDITION_HARMATTAN
#include <QNetworkConfigurationManager>
#include <QNetworkSession>
#include <meegotouchevents/meventfeed.h>
#endif
#ifdef Q_WS_MAEMO_5
#include "alarm.h"
#endif

using namespace QtJson;

Events::Events(QObject *parent) :
    QObject(parent),
    m_nam(new QNetworkAccessManager(this)),
    m_database(new Database(this)),
    m_youtube(new YouTube(this)),
    m_dailymotion(new Dailymotion(this)),
    m_vimeo(new Vimeo(this)),
    m_index(0),
    m_initialized(false)
{
}

void Events::init() {
    m_youtube->setNetworkAccessManager(m_nam);
    m_dailymotion->setNetworkAccessManager(m_nam);
    m_vimeo->setNetworkAccessManager(m_nam);
#ifdef MEEGO_EDITION_HARMATTAN
    QNetworkConfigurationManager configManager;
    QNetworkSession *session = new QNetworkSession(configManager.defaultConfiguration(), this);
    session->setSessionProperty("ConnectInBackground", true);
    this->connect(session, SIGNAL(error(QNetworkSession::SessionError)), this, SLOT(syncFailed()));
    this->connect(this, SIGNAL(destroyed()), session, SLOT(close()));
    session->open();
#endif
    this->connect(m_database, SIGNAL(gotYouTubeAccount(QString,QString,QString)), m_youtube, SLOT(setAccount(QString,QString,QString)));
    this->connect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), m_database, SLOT(setYouTubeAccessToken(QString)));
    this->connect(m_database, SIGNAL(gotDailymotionAccount(QString,QString,QString)), m_dailymotion, SLOT(setAccount(QString,QString,QString)));
    this->connect(m_dailymotion, SIGNAL(accessTokenRefreshed(QString,QString)), m_database, SLOT(setDailymotionAccessToken(QString,QString)));
    this->connect(m_database, SIGNAL(gotVimeoAccount(QString,QString,QString)), m_vimeo, SLOT(setAccount(QString,QString,QString)));
    this->connect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getYouTubeEvents()));
    this->connect(m_dailymotion, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getDailymotionEvents()));
    this->connect(m_youtube, SIGNAL(refreshError()), this, SLOT(syncFailed()));
    this->connect(m_dailymotion, SIGNAL(refreshError()), this, SLOT(syncFailed()));
    this->connect(m_database, SIGNAL(error(QString)), this, SLOT(syncFailed()));

    m_database->restoreAccounts();
    m_initialized = true;
}

bool Events::enableSync() {
    QSettings settings("cuteTube", "cuteTube");
#ifdef MEEGO_EDITION_HARMATTAN
    settings.setValue("Events/eventFeedEnabled", true);
    bool success = true;
#else
    qint64 cookie = add_alarm();
    bool success = cookie > 0;

    if (success) {
        settings.setValue("Events/eventsAlarmCookie", cookie);
        settings.setValue("Events/eventFeedEnabled", true);
        QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");
        iface.call(QDBus::NoBlock, "addRefreshAction", "cutetube", "dbus:session com.maemo.cuteTubeEvents / com.maemo.cuteTubeEvents startSync");
    }
#endif
    if (success) {
        this->startSync();
    }

    return success;
}

bool Events::disableSync() {
    QSettings settings("cuteTube", "cuteTube");
#ifdef MEEGO_EDITION_HARMATTAN
    settings.remove("Events/youtubeLatest");
    settings.remove("Events/dailymotionLatest");
    settings.remove("Events/vimeoLatest");
    settings.setValue("Events/eventFeedEnabled", false);
    MEventFeed::instance()->removeItemsBySourceName("cutetube");
    this->clearThumbnails();
    bool success = true;
#else
    qint64 cookie = settings.value("Events/eventsAlarmCookie", 0).toLongLong();
    bool success = cookie > 0;

    if (success) {
        success = delete_alarm(cookie);

        if (success) {
            settings.remove("Events/eventsAlarmCookie");
            settings.remove("Events/youtubeLatest");
            settings.remove("Events/dailymotionLatest");
            settings.remove("Events/vimeoLatest");
            settings.setValue("Events/eventFeedEnabled", false);
            QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");
            iface.call(QDBus::NoBlock, "removeEventsBySourceName", "cutetube");
            iface.call(QDBus::NoBlock, "removeRefreshAction", "cutetube");
        }
    }
#endif
    QTimer::singleShot(1000, QCoreApplication::instance(), SLOT(quit()));

    return success;
}

void Events::startSync() {
    if (!m_initialized) {
        this->init();
    }

    this->getEvents();
}

void Events::abortSync() {
    QCoreApplication::instance()->quit();
}

void Events::syncSuccess() {
    QCoreApplication::instance()->quit();
}

void Events::syncFailed() {
    QCoreApplication::instance()->quit();
}

void Events::getEvents() {
    if (m_youtube->userSignedIn()) {
        this->getYouTubeEvents();
    }
    else if (m_dailymotion->userSignedIn()) {
        this->getDailymotionEvents();
    }
    else if (m_vimeo->userSignedIn()) {
        this->getVimeoEvents();
    }
}

void Events::getYouTubeEvents() {
    QDateTime youtubeLatest = QDateTime::fromMSecsSinceEpoch(QSettings("cuteTube", "cuteTube").value("Events/youtubeLatest", 0).toLongLong());
    QNetworkReply *reply = m_youtube->createReply(QString("https://gdata.youtube.com/feeds/api/users/default/newsubscriptionvideos?v=2.1&max-results=50&fields=openSearch:totalResults,entry[xs:dateTime(published) > xs:dateTime('%1')](id,media:group(media:title,media:thumbnail[@yt:name='default'or@yt:name='hqdefault'],media:description,media:player[@url],media:credit,yt:uploaded,yt:uploaderId,yt:videoid))").arg(youtubeLatest.toString(Qt::ISODate)));
    this->connect(reply, SIGNAL(finished()), this, SLOT(processYouTubeEvents()));
}

void Events::processYouTubeEvents() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->syncFailed();
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        m_youtube->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            QDomNode entry = entries.at(i);
            QDomElement mediaGroup = entry.firstChildElement("media:group");
            Event event;
            event.setId(mediaGroup.firstChildElement("yt:videoid").text());
            event.setSourceName("cutetube");
            event.setSourceDisplayName("cuteTube");
            event.setIconPath("/usr/share/icons/hicolor/64x64/apps/cutetube64.png");
            event.setTitle(mediaGroup.firstChildElement("media:title").text());
            event.setBody(mediaGroup.firstChildElement("media:description").text());
            event.setImagePathOne(mediaGroup.firstChildElement("media:thumbnail").attribute("url").section('/', 0, -2).append("/mqdefault.jpg"));
            event.setFooter(mediaGroup.firstChildElement("media:credit").attribute("yt:display"));
#ifdef MEEGO_EDITION_HARMATTAN
            event.setTimestamp(QDateTime::fromString(mediaGroup.firstChildElement("yt:uploaded").text(), Qt::ISODate).toLocalTime().toString("yyyy-MM-dd hh:mm:ss"));
            event.setAction(QString("com.maemo.cuteTube / com.maemo.cuteTube display %1").arg(this->base64SerializedVariant(QVariant("http://youtu.be/" + event.id()))));
#else
            event.setTimestamp(mediaGroup.firstChildElement("yt:uploaded").text());
            event.setAction("dbus:session com.maemo.cuteTube / com.maemo.cuteTube display http://youtu.be/" + event.id());
#endif
            m_events.append(event);

            if (i == 0) {
                QSettings("cuteTube", "cuteTube").setValue("Events/youtubeLatest", QDateTime::fromString(event.timestamp(), Qt::ISODate).toMSecsSinceEpoch());
            }
        }

        if (m_dailymotion->userSignedIn()) {
            this->getDailymotionEvents();
        }
        else {
#ifdef MEEGO_EDITION_HARMATTAN
            this->downloadThumbnails();
#else
            this->addEvents();
#endif
        }
    }

    reply->deleteLater();
}

void Events::getDailymotionEvents() {
    QNetworkReply *reply = m_dailymotion->createReply("https://api.dailymotion.com/me/subscriptions?limit=50&fields=created_time,description,id,owner.username,owner.id,thumbnail_medium_url,thumbnail_large_url,title,url");
    this->connect(reply, SIGNAL(finished()), this, SLOT(processDailymotionEvents()));
}

void Events::processDailymotionEvents() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->syncFailed();
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        m_dailymotion->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->syncFailed();
            return;
        }

        qint64 dailymotionLatest = QSettings("cuteTube", "cuteTube").value("Events/dailymotionLatest", 0).toLongLong();
        QVariantList entries = res.value("list").toList();

        for (int i = 0; i < entries.size(); i++) {
            QVariantMap entry = entries.at(i).toMap();
            qint64 date = entry.value("created_time").toLongLong() * 1000;

            if (date > dailymotionLatest) {
                Event event;
                event.setId(entry.value("id").toString());
                event.setSourceName("cutetube");
                event.setSourceDisplayName("cuteTube");
                event.setIconPath("/usr/share/icons/hicolor/64x64/apps/cutetube64.png");
                event.setTitle(entry.value("title").toString());
                event.setBody(entry.value("description").toString());
                event.setImagePathOne(entry.value("thumbnail_large_url").toString());
                event.setFooter(entry.value("owner.username").toString());
#ifdef MEEGO_EDITION_HARMATTAN
                event.setTimestamp(QDateTime::fromMSecsSinceEpoch(date).toLocalTime().toString("yyyy-MM-dd hh:mm:ss"));
                event.setAction(QString("com.maemo.cuteTube / com.maemo.cuteTube display %1").arg(this->base64SerializedVariant(QVariant("http://dai.ly/" + event.id()))));
#else
                event.setTimestamp(QString::number(date));
                event.setAction("dbus:session com.maemo.cuteTube / com.maemo.cuteTube display http://dai.ly/" + event.id());
#endif
                m_events.append(event);
            }
        }
        
        if (!entries.isEmpty()) {
            QSettings("cuteTube", "cuteTube").setValue("Events/dailymotionLatest", entries.first().toMap().value("created_time").toLongLong() * 1000);
        }

        if (m_vimeo->userSignedIn()) {
            this->getVimeoEvents();
        }
        else {
#ifdef MEEGO_EDITION_HARMATTAN
            this->downloadThumbnails();
#else
            this->addEvents();
#endif
        }
    }

    reply->deleteLater();
}

void Events::getVimeoEvents() {
    QNetworkReply *reply = m_vimeo->createReply("http://vimeo.com/api/rest/v2?method=vimeo.videos.getSubscriptions&per_page=50&full_response=true");
    this->connect(reply, SIGNAL(finished()), this, SLOT(processVimeoEvents()));
}

void Events::processVimeoEvents() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->syncFailed();
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDateTime vimeoLatest = QDateTime::fromMSecsSinceEpoch(QSettings("cuteTube", "cuteTube").value("Events/vimeoLatest", 0).toLongLong());
        QDomNodeList entries = doc.elementsByTagName("video");

        for (int i = 0; i < entries.size(); i++) {
            QDomElement entry = entries.at(i).toElement();
            QDateTime date = QDateTime::fromString(entry.firstChildElement("upload_date").text(), Qt::ISODate);

            if (date > vimeoLatest) {
                Event event;
                event.setId(entry.attribute("id"));
                event.setSourceName("cutetube");
                event.setSourceDisplayName("cuteTube");
                event.setIconPath("/usr/share/icons/hicolor/64x64/apps/cutetube64.png");
                event.setTitle(entry.firstChildElement("title").text());
                event.setBody(entry.firstChildElement("description").text());
                event.setFooter(entry.firstChildElement("owner").attribute("display_name"));

                QDomNodeList thumbnails = entry.firstChildElement("thumbnails").childNodes();

                if (thumbnails.size() > 2) {
                    event.setImagePathOne(thumbnails.at(2).toElement().text());
                }

#ifdef MEEGO_EDITION_HARMATTAN
                event.setTimestamp(date.toLocalTime().toString("yyyy-MM-dd hh:mm:ss"));
                event.setAction(QString("com.maemo.cuteTube / com.maemo.cuteTube display %1").arg(this->base64SerializedVariant(QVariant("http://vimeo.com/" + event.id()))));
#else
                event.setTimestamp(QString::number(date.toMSecsSinceEpoch()));
                event.setAction("dbus:session com.maemo.cuteTube / com.maemo.cuteTube display http://vimeo.com/" + event.id());
#endif
                m_events.append(event);
            }
        }

        if (!entries.isEmpty()) {
            QSettings("cuteTube", "cuteTube").setValue("Events/vimeoLatest", QDateTime::currentMSecsSinceEpoch());
        }
#ifdef MEEGO_EDITION_HARMATTAN
        this->downloadThumbnails();
#else
        this->addEvents();
#endif
    }
    else {
        this->syncFailed();
    }

    reply->deleteLater();
}

void Events::addEvents() {
#ifdef MEEGO_EDITION_HARMATTAN
    while (!m_events.isEmpty()) {
        QVariantList args;
        QVariantMap itemArgs;
        Event event = m_events.takeFirst();
        itemArgs.insert("title", event.title());
        itemArgs.insert("icon", event.iconPath());
        itemArgs.insert("body", event.body());
        itemArgs.insert("imageList", QStringList(event.imagePathOne()));
        itemArgs.insert("timestamp", event.timestamp());
        itemArgs.insert("video", false);
        itemArgs.insert("footer", event.footer());
        itemArgs.insert("sourceName", QString("cutetube"));
        itemArgs.insert("sourceDisplayName", QString("cuteTube"));
        itemArgs.insert("action", event.action());
        args.append(itemArgs);

        QDBusMessage message = QDBusMessage::createMethodCall(
                    "com.nokia.home.EventFeed",
                    "/eventfeed",
                    "com.nokia.home.EventFeed",
                    "addItem");
        message.setArguments(args);

        QDBusConnection bus = QDBusConnection::sessionBus();
        bus.callWithCallback(message, this, SLOT(dbusRequestCompleted(QDBusMessage)), SLOT(dbusError(QDBusError,QDBusMessage)));
    }
#else
    while (!m_events.isEmpty()) {
        QVariantList args;
        Event event = m_events.takeFirst();
        args.append(event.sourceName());
        args.append(event.sourceDisplayName());
        args.append(event.iconPath());
        args.append(event.title());
        args.append(event.body());
        args.append(QStringList(event.imagePathOne()));
        args.append(event.footer());
        args.append(event.timestamp());
        args.append(event.action());

        QDBusInterface("com.maemo.eventFeed", "/", "com.maemo.eventFeed").callWithArgumentList(QDBus::Block, "addEvent", args);
    }
#endif
    this->syncSuccess();
}

#ifdef MEEGO_EDITION_HARMATTAN
void Events::dbusRequestCompleted(const QDBusMessage &message) {
    if (message.arguments().first().toInt() < 0) {
        this->syncFailed();
    }
}

void Events::dbusError(const QDBusError &error, const QDBusMessage &message) {
    Q_UNUSED(error)
    Q_UNUSED(message)

    this->syncFailed();
}

QString Events::base64SerializedVariant(const QVariant &value) const
{
    QByteArray ba;
    QDataStream stream(&ba, QIODevice::WriteOnly);
    stream << value;
    return ba.toBase64();
}

void Events::downloadThumbnails() {
    if ((m_index >= 0) && (m_index < m_events.size())) {
        QFileInfo info(QString("/home/user/cuteTube/.feedthumbnails/%1.jpg").arg(m_events[m_index].id()));

        if (info.exists()) {
            m_events[m_index].setImagePathOne(info.absoluteFilePath());
            m_index++;
            this->downloadThumbnails();
        }
        else {
            QUrl url(m_events[m_index].imagePathOne());
            QNetworkRequest request(url);
            QNetworkReply *reply = m_nam->get(request);
            this->connect(reply, SIGNAL(finished()), this, SLOT(onThumbnailDownloaded()));
        }
    }
    else {
        this->addEvents();
    }
}

void Events::onThumbnailDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if ((!reply) || (m_index < 0) || (m_index >= m_events.size())) {
        this->syncFailed();
        return;
    }

    QDir dir("/home/user/cuteTube/.feedthumbnails/");
    dir.mkpath(dir.path());
    QFile file(QString("/home/user/cuteTube/.feedthumbnails/%1.jpg").arg(m_events[m_index].id()));

    if (file.open(QIODevice::WriteOnly)) {
        file.write(reply->readAll());
        file.close();
        m_events[m_index].setImagePathOne(file.fileName());
    }

    reply->deleteLater();
    m_index++;
    this->downloadThumbnails();
}

void Events::clearThumbnails() {
    QDir dir("/home/user/cuteTube/.feedthumbnails/");

    foreach (QString thumbnail, dir.entryList(QStringList("*.jpg"))) {
        dir.remove(thumbnail);
    }
}
#endif
