#include "urlgrabber.h"
#include "cookiejar.h"
#include "json.h"
#include "utils.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QDebug>

using namespace QtJson;

UrlGrabber* grabberInstance = 0;

UrlGrabber::UrlGrabber(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_format(Videos::Unknown),
    m_busy(false),
    m_cancelled(false)
{
    if (!grabberInstance) {
        grabberInstance = this;
    }
}

UrlGrabber* UrlGrabber::instance() {
    return grabberInstance;
}

void UrlGrabber::setYouTubeFormats(QSet<int> formats) {
    m_youtubeFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::LQ:
            videoFormat.value = 18;
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal (upto 360P)"));
            break;
        case Videos::Normal:
            videoFormat.value = 34;
            videoFormat.displayName = "360P FLV";
            break;
        case Videos::HQ:
            videoFormat.value = 35;
            videoFormat.displayName = "480P FLV";
            break;
        case Videos::HD:
            videoFormat.value = 22;
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = 37;
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setYouTubeFormats(): Invalid video format";
            break;
        }

        m_youtubeFormats.append(videoFormat);
    }
}

void UrlGrabber::setDailymotionFormats(QSet<int> formats) {
    m_dailymotionFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::Normal:
            videoFormat.value = "stream_h264_url";
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal"));
            break;
        case Videos::HQ:
            videoFormat.value = "stream_h264_hq_url";
            videoFormat.displayName = "480P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "stream_h264_hd_url";
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = "stream_h264_hd1080_url";
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setDailymotionFormats(): Invalid video format";
            break;
        }

        m_dailymotionFormats.append(videoFormat);
    }
}

void UrlGrabber::setVimeoFormats(QSet<int> formats) {
    m_vimeoFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::LQ:
            videoFormat.value = "mobile";
            videoFormat.displayName = tr("Mobile");
            break;
        case Videos::Normal:
            videoFormat.value = "sd";
            videoFormat.displayName = "360P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "hd";
            videoFormat.displayName = "720P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setVimeoFormats(): Invalid video format";
            break;
        }

        m_vimeoFormats.append(videoFormat);
    }
}

void UrlGrabber::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void UrlGrabber::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    emit currentOperationCancelled();
}

void UrlGrabber::getVideoUrl(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getYouTubeVideoUrl(id);
        break;
    case Services::Dailymotion:
        this->getDailymotionVideoUrl(id);
        break;
    case Services::Vimeo:
        this->getVimeoVideoUrl(id);
        break;
    default:
        qWarning() << "UrlGrabber::getVideoUrl(): No/invalid service specied";
        return;
    }
}

QString UrlGrabber::decryptYouTubeSignature(const QString &s) {
    QString signature;

    switch (s.size()) {
    case 88: {
        signature = s.at(48)
                + Utils::reverseString(s.mid(68, 14))
                + s.at(82)
                + Utils::reverseString(s.mid(63, 4))
                + s.at(85)
                + Utils::reverseString(s.mid(49, 13))
                + s.at(67)
                + Utils::reverseString(s.mid(13, 35))
                + s.at(3)
                + Utils::reverseString(s.mid(4, 8))
                + s.at(2)
                + s.at(12);
    }
        break;
    case 87: {
        signature = s.at(62)
                + Utils::reverseString(s.mid(63, 20))
                + s.at(83)
                + Utils::reverseString(s.mid(53, 9))
                + s.at(0)
                + Utils::reverseString(s.mid(3, 49));
    }
        break;
    case 86: {
        signature = s.mid(2, 61)
                + s.at(82)
                + s.mid(64, 18)
                + s.at(63);
    }
        break;
    case 85: {
        signature = s.at(76)
                + Utils::reverseString(s.mid(77, 6))
                + s.at(83)
                + Utils::reverseString(s.mid(61, 15))
                + s.at(0)
                + Utils::reverseString(s.mid(51, 9))
                + s.at(1)
                + Utils::reverseString(s.mid(3, 47));
    }
        break;
    case 84: {
        signature = Utils::reverseString(s.mid(37, 47))
                + s.at(2)
                + Utils::reverseString(s.mid(27, 9))
                + s.at(3)
                + Utils::reverseString(s.mid(4, 22))
                + s.at(26);
    }
        break;
    case 83: {
        signature = s.at(52)
                + Utils::reverseString(s.mid(56, 26))
                + s.at(2)
                + Utils::reverseString(s.mid(53, 2))
                + s.at(82)
                + Utils::reverseString(s.mid(37, 15))
                + s.at(55)
                + Utils::reverseString(s.mid(3, 33))
                + s.at(36);
    }
        break;
    case 82: {
        signature = s.at(36)
                + Utils::reverseString(s.mid(68, 13))
                + s.at(81)
                + Utils::reverseString(s.mid(41, 26))
                + s.at(33)
                + Utils::reverseString(s.mid(37, 3))
                + s.at(40)
                + s.at(35)
                + s.at(0)
                + s.at(67)
                + Utils::reverseString(s.mid(1, 32))
                + s.at(34);
    }
        break;
    default:
        break;
    }

    return signature;
}

void UrlGrabber::getYouTubeVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("https://www.youtube.com/watch");
    url.addQueryItem("v", id);
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    url.addQueryItem("has_verified", "1");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseYouTubeVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseYouTubeVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << response;

        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : message);
    }
    else {
        response = response.section("url_encoded_fmt_stream_map\":", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&");
        int unescapes = 0;

        while ((response.contains('%')) && (unescapes < 10)) {
            response = QByteArray::fromPercentEncoding(response.toUtf8());
            unescapes++;
        }

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            response = response.replace(QRegExp("[&,\"]s="), "&signature=");
        }
        else {
            response = response.replace(QRegExp("[&,\"]sig="), "&signature=");
        }

        QStringList urlStrings = response.split("url=", QString::SkipEmptyParts);
        QStringList sigs = response.split("signature=");
        QStringList signatures;
        QMap<int, QString> formats;

        if (encryptedSignatures) {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << this->decryptYouTubeSignature(sigs.at(i).section(QRegExp("(&|,)"), 0, 0));
            }
        }
        else {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
            }
        }

        foreach (QString urlString, urlStrings) {
            if (urlString.startsWith("http")) {
                QUrl url(urlString.section(QRegExp("[&,\"]itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();

                if (key > 0) {
                    if (!signatures.isEmpty()) {
                        url.removeQueryItem("signature");
                        url.addQueryItem("signature", signatures.takeFirst());
                    }

                    formats[key] = url.toString();
                }
            }
        }

        QString videoUrl;
        int index = 0;

        while ((videoUrl.isEmpty()) && (index < m_youtubeFormats.size())) {
            videoUrl = formats.value(m_youtubeFormats.at(index).value.toInt(), "");
            m_format = m_youtubeFormats.at(index).format;
            index++;
        }

        if (!videoUrl.startsWith("http")) {
            qDebug() << response;

            QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
            emit error(message.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : message);
        }
        else {
            emit gotVideoUrl(QUrl(videoUrl), m_format);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://www.dailymotion.com/embed/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseDailymotionVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantMap info = Json::parse(response.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QUrl url;
    int i = 0;

    while ((url.isEmpty()) && (i < m_dailymotionFormats.size())) {
        url.setUrl(info.value(m_dailymotionFormats.at(i).value.toString(), "").toString());
        m_format = m_dailymotionFormats.at(i).format;
        i++;
    }

    if (url.isEmpty()) {
        qDebug() << response;
        QString errorString = info.value("error").toMap().value("message").toString();
        emit error(errorString.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : errorString);
    }
    else {
        emit gotVideoUrl(url, m_format);
    }
    
    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseVimeoVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    QVariantMap paramMap = Json::parse(params).toMap();

    if (paramMap.isEmpty()) {
        qDebug() << response;
        this->setBusy(false);
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");
        QString quality;

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QVariantList qualities = formatMap.value(codec).toList();
            int i = 0;

            while ((quality.isEmpty()) && (i < m_vimeoFormats.size())) {
                if (qualities.contains(m_vimeoFormats.at(i).value)) {
                    quality = m_vimeoFormats.at(i).value.toString();
                    m_format = m_vimeoFormats.at(i).format;
                }

                i++;
            }

            QString timeStamp = requestMap.value("timestamp").toString();
            QString signature = requestMap.value("signature").toString();
            QString id = videoMap.value("id").toString();
            QUrl url("http://player.vimeo.com/play_redirect");
            url.addQueryItem("quality", quality);
            url.addQueryItem("clip_id", id);
            url.addQueryItem("time", timeStamp);
            url.addQueryItem("sig", signature);

            if (quality != "mobile") {
                url.addQueryItem("codecs", codec);
            }

            this->getVimeoVideoRedirect(url);
        }
        else {
            this->setBusy(false);
            qDebug() << response;
            emit error(tr("Unable to retrieve video. Access may be restricted"));
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoRedirect()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoRedirect() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isEmpty()) {
        emit gotVideoUrl(reply->request().url(), m_format);
    }
    else {
        emit gotVideoUrl(redirect, m_format);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVideoFormats(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getAvailableYouTubeVideoFormats(id);
        break;
    case Services::Dailymotion:
        this->getAvailableDailymotionVideoFormats(id);
        break;
    case Services::Vimeo:
        this->getAvailableVimeoVideoFormats(id);
        break;
    default:
        return;
    }
}

void UrlGrabber::getAvailableYouTubeVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("https://www.youtube.com/watch");
    url.addQueryItem("v", id);
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    url.addQueryItem("has_verified", "1");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkYouTubeVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QVariantList formats;
    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << response;

        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve formats. Access may be restricted") : message);
    }
    else {
        response = response.section("url_encoded_fmt_stream_map\":", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&");
        int unescapes = 0;

        while ((response.contains('%')) && (unescapes < 10)) {
            response = QByteArray::fromPercentEncoding(response.toUtf8());
            unescapes++;
        }

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            response = response.replace(QRegExp("[&,\"]s="), "&signature=");
        }
        else {
            response = response.replace(QRegExp("[&,\"]sig="), "&signature=");
        }

        QStringList urlStrings = response.split("url=", QString::SkipEmptyParts);
        QStringList sigs = response.split("signature=");
        QStringList signatures;

        if (encryptedSignatures) {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << this->decryptYouTubeSignature(sigs.at(i).section(QRegExp("(&|,)"), 0, 0));
            }
        }
        else {
            for (int i = 1; i < sigs.size(); i++) {
                signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
            }
        }

        foreach (QString urlString, urlStrings) {
            if ((urlString.startsWith("http")) && (!signatures.isEmpty())) {
                QUrl url(urlString.section(QRegExp("[&,\"]itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();
                QVariantMap format;

                switch (key) {
                case 37:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "1080P AVC1";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
                case 22:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "720P AVC1";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
#ifndef SYMBIAN_OS
                case 35:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "480P FLV";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
                case 34:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = "360P AVC1";
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                    break;
#endif
                case 18:
                    url.removeQueryItem("signature");
                    url.addQueryItem("signature", signatures.takeFirst());
                    format["name"] = QString("%1 AVC1").arg(tr("Normal"));
                    format["service"] = Services::YouTube;
                    format["url"] = url;
                    formats << format;
                default:
                    signatures.takeFirst();
                    break;
                }
            }
        }
    }

    if (formats.isEmpty()) {
        qDebug() << response;

        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve formats. Access may be restricted") : message);
    }
    else {
        emit gotVideoFormats(formats);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableDailymotionVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://www.dailymotion.com/embed/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkDailymotionVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantMap info = Json::parse(response.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QVariantList formats;
    int i = 0;

    QMap<QString, QString> formatNames;
    formatNames["stream_h264_url"] = QString("%1 AVC1").arg(tr("Normal"));
    formatNames["stream_h264_hq_url"] = "480P AVC1";
    formatNames["stream_h264_hd_url"] = "720P AVC1";
    formatNames["stream_h264_hd1080_url"] = "1080P AVC1";

    foreach (QString formatName, formatNames.keys()) {
        QUrl url = info.value(formatName).toUrl();

        if (!url.isEmpty()) {
            QVariantMap format;
            format["name"] = formatNames.value(formatName);
            format["service"] = Services::Dailymotion;
            format["url"] = url;
            formats << format;
        }

        i++;
    }

    if (formats.isEmpty()) {
        qDebug() << response;
        QString errorString = info.value("error").toMap().value("message").toString();
        emit error(errorString.isEmpty() ? tr("Unable to retrieve video formats. Access may be restricted") : errorString);
    }
    else {
        emit gotVideoFormats(formats);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVimeoVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    QVariantMap paramMap = Json::parse(params).toMap();

    if (paramMap.isEmpty()) {
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QStringList qualities = formatMap.value(codec).toStringList();
            QVariantList formats;
            QMap<QString, QString> formatNames;
            formatNames["mobile"] = tr("Mobile");
            formatNames["sd"] = "360P AVC1";
            formatNames["hd"] = "720P AVC1";

            while (!qualities.isEmpty()) {
                QString quality = qualities.takeFirst();
                QString timeStamp = requestMap.value("timestamp").toString();
                QString signature = requestMap.value("signature").toString();
                QString id = videoMap.value("id").toString();

                QVariantMap format;
                format["name"] = formatNames[quality];
                format["service"] = Services::Vimeo;
                QUrl url("http://player.vimeo.com/play_redirect");
                url.addQueryItem("quality", quality);
                url.addQueryItem("clip_id", id);
                url.addQueryItem("time", timeStamp);
                url.addQueryItem("sig", signature);

                if (quality != "mobile") {
                    url.addQueryItem("codecs", codec);
                }

                format["url"] = url;
                formats << format;
            }

            emit gotVideoFormats(formats);
        }
        else {
            qDebug() << response;
            emit error(tr("Unable to retrieve video formats. Access may be restricted"));
        }
    }

    reply->deleteLater();
}
