#include "youtubevideodescriptiontab.h"
#include "youtubeplaylistdialog.h"
#include "youtubemetadatadialog.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "youtubevideolistwindow.h"
#include "youtubechannelwindow.h"
#include "videoplaybackwindow.h"
#include "../base/youtube.h"
#include "../base/thumbnailcache.h"
#include "../base/windowstack.h"
#include "../base/urlgrabber.h"
#include "../base/settings.h"
#include "../base/sharing.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "thumbnaillabel.h"
#include <QAction>
#include <QLabel>
#include <QPushButton>
#include <QScrollArea>
#include <QGridLayout>

YouTubeVideoDescriptionTab::YouTubeVideoDescriptionTab(QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_video(video),
    m_editAction(new QAction(tr("Edit"), this)),
    m_favouritesAction(new QAction(video->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_shareAction(new QAction(tr("Share"), this)),
    m_likeButton(new QPushButton(QIcon(":/images/like.png"), "", this)),
    m_dislikeButton(new QPushButton(QIcon(":/images/dislike.png"), "", this)),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_likesLabel(new QLabel(QString::number(m_video.data()->likes()), this)),
    m_dislikesLabel(new QLabel(QString::number(m_video.data()->dislikes()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Views")).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_editAction);
    this->addAction(m_favouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_watchLaterAction);
    this->addAction(m_queueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);
    this->addAction(m_shareAction);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_likesLabel->setFont(smallFont);
    m_dislikesLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_likeButton->setFixedSize(70, 70);
    m_likeButton->setEnabled((YouTube::instance()->userSignedIn()) && (m_video.data()->artistId() != YouTube::instance()->username()));
    m_dislikeButton->setFixedSize(70, 70);
    m_dislikeButton->setEnabled(m_likeButton->isEnabled());
    m_likesLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);

    grid->addWidget(m_titleLabel, 0, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_likeButton, 2, 0);
    grid->addWidget(m_likesLabel, 2, 1);
    grid->addWidget(m_dislikeButton, 2, 2);
    grid->addWidget(m_dislikesLabel, 2, 3);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 4, Qt::AlignTop);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_likeButton, SIGNAL(clicked()), this, SLOT(likeVideo()));
    this->connect(m_dislikeButton, SIGNAL(clicked()), this, SLOT(dislikeVideo()));
    this->connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    this->connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    this->connect(m_video.data(), SIGNAL(likesChanged(int)), this, SLOT(onLikesChanged(int)));
    this->connect(m_video.data(), SIGNAL(dislikesChanged(int)), this, SLOT(onDislikesChanged(int)));
    this->connect(m_video.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));
    this->connect(m_editAction, SIGNAL(triggered()), this, SLOT(showEditMetadataDialog()));
    this->connect(m_favouritesAction, SIGNAL(triggered()), this, SLOT(toggleFavourite()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(showPlaylistDialog()));
    this->connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideoToWatchLaterPlaylist()));
    this->connect(m_queueAction, SIGNAL(triggered()), this, SLOT(addVideoToPlaybackQueue()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownload()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownload()));
    this->connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));

    this->setMenuActions();
    this->setThumbnails();
    this->onOrientationChanged();
}

void YouTubeVideoDescriptionTab::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void YouTubeVideoDescriptionTab::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

void YouTubeVideoDescriptionTab::setMenuActions() {
    bool signedIn = YouTube::instance()->userSignedIn();

    m_editAction->setEnabled((signedIn) && (m_video.data()->artistId() == YouTube::instance()->username()));
    m_favouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
}

void YouTubeVideoDescriptionTab::setThumbnails() {
    QImage thumbnail = ThumbnailCache::instance()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = ThumbnailCache::instance()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void YouTubeVideoDescriptionTab::showEditMetadataDialog() {
    YouTubeMetadataDialog *dialog = new YouTubeMetadataDialog(m_video, this);
    dialog->open();
}

void YouTubeVideoDescriptionTab::toggleFavourite() {
    if (m_video.data()->favourite()) {
        YouTube::instance()->deleteFromFavourites(QStringList(m_video.data()->favouriteId()));
    }
    else {
        YouTube::instance()->addToFavourites(QStringList(m_video.data()->videoId()));
    }
}

void YouTubeVideoDescriptionTab::onFavouriteChanged(bool favourite) {
    m_favouritesAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void YouTubeVideoDescriptionTab::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void YouTubeVideoDescriptionTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void YouTubeVideoDescriptionTab::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        this->showFacebookDialog();
        return;
    case Services::Twitter:
        this->showTwitterDialog();
        return;
    case Services::Email:
        Sharing::instance()->shareViaEmail(m_video.data()->title(), m_video.data()->url());
        return;
    case Services::Clipboard:
        Sharing::instance()->copyToClipboard(m_video.data()->url());
        return;
    default:
        qWarning() << "YouTubeVideoDescriptionTab::shareVideo(): Invalid service.";
        return;
    }
}

void YouTubeVideoDescriptionTab::showFacebookDialog() {
    ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_video, this);
    dialog->open();
}

void YouTubeVideoDescriptionTab::showTwitterDialog() {
    ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_video, this);
    dialog->open();
}

void YouTubeVideoDescriptionTab::addVideoToPlaylist(const QString &id) {
    YouTube::instance()->addToPlaylist(QStringList(m_video.data()->videoId()), id);
}

void YouTubeVideoDescriptionTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    YouTube::instance()->createPlaylist(playlist, QStringList(m_video.data()->videoId()));
}

void YouTubeVideoDescriptionTab::addVideoToWatchLaterPlaylist() {
    YouTube::instance()->addToWatchLaterPlaylist(QStringList(m_video.data()->videoId()));
}

void YouTubeVideoDescriptionTab::addVideoToPlaybackQueue() {
    PlaybackListModel::playbackQueue()->addVideo(m_video);
}

void YouTubeVideoDescriptionTab::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_video);
}

void YouTubeVideoDescriptionTab::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_video, true);
}

void YouTubeVideoDescriptionTab::likeVideo() {
    if ((!m_video.data()->liked()) && (!m_video.data()->disliked())) {
        YouTube::instance()->likeVideo(m_video.data()->videoId());
    }
}

void YouTubeVideoDescriptionTab::dislikeVideo() {
    if ((!m_video.data()->liked()) && (!m_video.data()->disliked())) {
        YouTube::instance()->dislikeVideo(m_video.data()->videoId());
    }
}

void YouTubeVideoDescriptionTab::onLikesChanged(int likes) {
    m_likesLabel->setText(QString::number(likes));
}

void YouTubeVideoDescriptionTab::onDislikesChanged(int dislikes) {
    m_dislikesLabel->setText(QString::number(dislikes));
}

void YouTubeVideoDescriptionTab::playVideo() {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_video, WindowStack::instance()->currentWindow());
        window->show();
    }
    else {
        UrlGrabber::instance()->getYouTubeVideoUrl(m_video.data()->videoId());
    }
}

void YouTubeVideoDescriptionTab::showUserChannel() {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(m_video.data()->artistId(), WindowStack::instance()->currentWindow());
    window->show();
}

void YouTubeVideoDescriptionTab::search(const QString &query) {
    YouTubeVideoListWindow *window = new YouTubeVideoListWindow(WindowStack::instance()->currentWindow());
    window->search(query, Settings::instance()->searchOrder(), Settings::instance()->searchTime(), Settings::instance()->searchDuration(), Settings::instance()->languageFilter());
    window->show();
}
