#include "youtuberelatedvideostab.h"
#include "youtubeplaylistdialog.h"
#include "videolistdelegate.h"
#include "youtubevideoinfowindow.h"
#include "youtubevideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/youtube.h"
#include "../base/settings.h"
#include "../base/urlgrabber.h"
#include "../base/windowstack.h"
#include "../base/transfermanager.h"
#include "../base/playbacklistmodel.h"
#include "../base/notifications.h"
#include "../base/youtubevideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>

YouTubeRelatedVideosTab::YouTubeRelatedVideosTab(QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_video(video),
    m_model(new YouTubeVideoListModel(this)),
    m_listView(new ListView(this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_contextMenu(new QMenu(this)),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_watchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterPlaylist()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_addToFavouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_watchLaterAction);
    this->addAction(m_addToQueueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    this->connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideosToWatchLaterPlaylist()));
    this->connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    m_model->getRelatedVideos(m_video.data()->videoId());

    this->setMenuActions();
    this->onOrientationChanged();
}

void YouTubeRelatedVideosTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeRelatedVideosTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeRelatedVideosTab::setMenuActions() {
    bool signedIn = YouTube::instance()->userSignedIn();

    m_addToFavouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
    m_addToFavouritesContextAction->setVisible(signedIn);
    m_addToPlaylistContextAction->setVisible(signedIn);
    m_watchLaterContextAction->setVisible(signedIn);
}

void YouTubeRelatedVideosTab::onLoadingChanged(bool loading) {
    Notifications::toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void YouTubeRelatedVideosTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void YouTubeRelatedVideosTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeRelatedVideosTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(video, WindowStack::instance()->currentWindow());
    window->show();
}

void YouTubeRelatedVideosTab::playVideo(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), WindowStack::instance()->currentWindow());
        window->show();
    }
    else {
        UrlGrabber::instance()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}

void YouTubeRelatedVideosTab::addVideosToFavourites() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addVideosToPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addVideosToWatchLaterPlaylist() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_model, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addVideoDownloads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_model, tr("Download"), SLOT(addVideoDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addAudioDownloads() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_model, tr("Download"), SLOT(addAudioDownloads()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::queueVideos() {
    YouTubeVideoSelectionWindow *window = new YouTubeVideoSelectionWindow(m_model, tr("Add to queue"), SLOT(queueVideos()), WindowStack::instance()->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToFavouritesContextAction);
}

void YouTubeRelatedVideosTab::addVideoToFavourites() {
    YouTube::instance()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeRelatedVideosTab::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void YouTubeRelatedVideosTab::addVideoToPlaylist(const QString &id) {
    YouTube::instance()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void YouTubeRelatedVideosTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    YouTube::instance()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeRelatedVideosTab::addVideoToWatchLaterPlaylist() {
    YouTube::instance()->addToWatchLaterPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void YouTubeRelatedVideosTab::addVideoDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeRelatedVideosTab::addAudioDownload() {
    TransferManager::instance()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void YouTubeRelatedVideosTab::queueVideo() {
    PlaybackListModel::playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}
