#include "archivewindow.h"
#include "archivevideolistdelegate.h"
#include "querydialog.h"
#include "videoplaybackwindow.h"
#include "listview.h"
#include "filterbox.h"
#include "../base/archivelistmodel.h"
#include "../base/settings.h"
#include "../base/videolauncher.h"
#include <QMenu>
#include <QAction>
#include <QLabel>
#include <QVBoxLayout>
#include <QMenuBar>

ArchiveWindow::ArchiveWindow( StackedWindow *parent) :
    StackedWindow(parent),
    m_model(new ArchiveListModel(this)),
    m_listView(new ListView(this)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_deleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(showConfirmDeleteDialog()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_vbox(0)
{
    this->setWindowTitle(tr("Archive"));
    this->setCentralWidget(new QWidget());

    m_vbox = new QVBoxLayout(this->centralWidget());
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_listView);
    m_vbox->addWidget(m_filterBox);
    m_filterBox->hide();
    m_noResultsLabel->hide();
    m_listView->setModel(m_model);
    m_listView->setRootIndex(m_model->rootIndex());
    m_listView->setItemDelegate(new ArchiveVideoListDelegate(m_listView));

    this->connect(m_model, SIGNAL(directoryLoaded(QString)), this, SLOT(onDirectoryLoaded()));
    this->connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));

    this->onOrientationChanged();
}

void ArchiveWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void ArchiveWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void ArchiveWindow::onDirectoryLoaded() {
    m_noResultsLabel->setVisible(m_model->rowCount() == 0);
}

void ArchiveWindow::onItemClicked(const QModelIndex &index) {
    if (Settings::instance()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(QList< QSharedPointer<VideoItem> >() << m_model->get(index), this);
        window->show();
    }
    else {
        VideoLauncher::launchVideo(index.data(ArchiveListModel::FilePathRole).toUrl());
    }
}

void ArchiveWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteAction);
}

void ArchiveWindow::showConfirmDeleteDialog() {
    QueryDialog *dialog = new QueryDialog(tr("Do you wish to delete '%1'?").arg(m_listView->currentIndex().data(ArchiveListModel::TitleRole).toString()), this);
    dialog->setWindowTitle(tr("Delete?"));
    dialog->open();

    this->connect(dialog, SIGNAL(accepted()), this, SLOT(deleteVideo()));
}

void ArchiveWindow::deleteVideo() {
    m_model->deleteVideo(m_listView->currentIndex());
}

void ArchiveWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        this->showFilterBox();
        m_filterBox->setFilterText(event->text());
    }
}

void ArchiveWindow::showFilterBox() {
    m_filterBox->show();
    m_filterBox->setFocus(Qt::OtherFocusReason);
}
