#include "transfermanager.h"
#include "database.h"
#include "settings.h"

TransferManager* managerInstance = 0;

TransferManager::TransferManager(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_transfers(new QList< QSharedPointer<TransferItem> >)
{
    if (!managerInstance) {
        managerInstance = this;
    }
}

TransferManager::~TransferManager() {
    m_transfers->clear();
    delete m_transfers;
    m_transfers = 0;
}

TransferManager* TransferManager::instance() {
    return managerInstance;
}

void TransferManager::restoreStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers = Database::instance()->getStoredDownloads();

    while (!transfers.isEmpty()) {
        QSharedPointer<TransferItem> transfer = transfers.takeFirst();
        transfer.data()->setNetworkAccessManager(this->networkAccessManager());
        transfer.data()->setStatus(Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused);
        this->transferList()->append(transfer);
        this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
        this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferPriorityChanged(Transfers::TransferPriority)));
        this->connect(transfer.data(), SIGNAL(videoFormatChanged(Videos::VideoFormat)), this, SLOT(onTransferVideoFormatChanged(Videos::VideoFormat)));
        this->connect(transfer.data(), SIGNAL(saveAsAudioChanged(bool)), this, SLOT(onTransferSaveAsAudioChanged(bool)));
        this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged(QString)));
        this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged(qint64)));

        if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
            transfer.data()->startTransfer();
        }
    }
}

QString TransferManager::generateTransferId(QString seedFileName) const {
    return QString("%1_%2").arg(QString::number(QDateTime::currentMSecsSinceEpoch())).arg(seedFileName);
}

void TransferManager::addDownloadTransfer(VideoItem *video, bool saveAsAudio, bool notify) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(this->generateTransferId(video->videoId()), video, Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused, saveAsAudio));
    transfer.data()->setDownloadPath(Settings::instance()->downloadPath() + ".incomplete/");
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));
    this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferPriorityChanged(Transfers::TransferPriority)));
    this->connect(transfer.data(), SIGNAL(videoFormatChanged(Videos::VideoFormat)), this, SLOT(onTransferVideoFormatChanged(Videos::VideoFormat)));
    this->connect(transfer.data(), SIGNAL(saveAsAudioChanged(bool)), this, SLOT(onTransferSaveAsAudioChanged(bool)));
    this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged(QString)));
    this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged(qint64)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    Database::instance()->storeDownload(transfer);

    if (notify) {
        emit alert(tr("Video download added to transfer queue"));
    }
}

void TransferManager::addDownloadTransfer(QSharedPointer<VideoItem> video, bool saveAsAudio, bool notify) {
    this->addDownloadTransfer(video.data(), saveAsAudio, notify);
}

void TransferManager::addDownloadTransfers(QList<VideoItem *> videos, bool saveAsAudio) {
    while (!videos.isEmpty()) {
        this->addDownloadTransfer(videos.takeFirst(), saveAsAudio, false);
    }

    emit alert(tr("Video download(s) added to transfer queue"));
}

void TransferManager::addDownloadTransfers(QList<QSharedPointer<VideoItem> > videos, bool saveAsAudio) {
    while (!videos.isEmpty()) {
        this->addDownloadTransfer(videos.takeFirst().data(), saveAsAudio, false);
    }

    emit alert(tr("Video download(s) added to transfer queue"));
}

void TransferManager::addUploadTransfer(const QVariantMap &metadata) {
    QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem(metadata, Settings::instance()->startTransfersAutomatically() ? Transfers::Queued : Transfers::Paused));
    transfer.data()->setNetworkAccessManager(this->networkAccessManager());
    this->transferList()->append(transfer);
    this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged(Transfers::TransferStatus)));

    if ((transfer.data()->status() == Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        transfer.data()->startTransfer();
    }

    emit alert(tr("Video upload added to transfer queue"));
}

QSharedPointer<TransferItem> TransferManager::getTransfer(TransferItem *transfer) const {
    for (int i = 0; i < this->transferList()->size(); i++) {
        if (this->transferList()->at(i).data() == transfer) {
            return this->transferList()->at(i);
        }
    }

    return QSharedPointer<TransferItem>();
}

QSharedPointer<TransferItem> TransferManager::getNextTransfer() const {
    int i = 0;
    int priority = Transfers::HighPriority;

    while (priority <= Transfers::LowPriority) {
        while (i < this->transferList()->size()) {
            if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
                if ((transfer.data()->priority() == priority) && (transfer.data()->status() == Transfers::Queued)) {
                    return transfer;
                }
            }

            i++;
        }

        priority++;
        i = 0;
    }

    return QSharedPointer<TransferItem>();
}

bool TransferManager::removeTransfer(QSharedPointer<TransferItem> transfer) {
    int row = this->transferList()->indexOf(transfer);

    if (row >= 0) {
        this->transferList()->removeAt(row);
        emit transferRemoved(row);
        return true;
    }

    return false;
}

void TransferManager::onTransferStatusChanged(Transfers::TransferStatus status) {
    if ((status <= Transfers::Queued) && (this->concurrentTransfers() < Settings::instance()->maximumConcurrentTransfers())) {
        if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
            transfer.data()->startTransfer();
        }
    }

    if ((status == Transfers::Completed) || (status == Transfers::Cancelled)) {
        if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
            if (status == Transfers::Completed) {
                this->onTransferCompleted(this->getTransfer(transfer));
            }
            else  {
                this->onTransferCancelled(this->getTransfer(transfer));
            }
        }
    }
}

void TransferManager::onTransferPriorityChanged(Transfers::TransferPriority priority) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
        QMetaObject::invokeMethod(Database::instance(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "priority"), Q_ARG(QVariant, priority));
    }
}

void TransferManager::onTransferVideoFormatChanged(Videos::VideoFormat videoFormat) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
        QMetaObject::invokeMethod(Database::instance(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "videoFormat"), Q_ARG(QVariant, videoFormat));
    }
}

void TransferManager::onTransferSaveAsAudioChanged(bool saveAsAudio) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
        QMetaObject::invokeMethod(Database::instance(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "saveAsAudio"), Q_ARG(QVariant, saveAsAudio));
    }
}

void TransferManager::onTransferFileNameChanged(const QString &fileName) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
        QMetaObject::invokeMethod(Database::instance(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "fileName"), Q_ARG(QVariant, fileName));
    }
}

void TransferManager::onTransferSizeChanged(qint64 size) {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(this->sender())) {
        QMetaObject::invokeMethod(Database::instance(), "updateStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer->id()), Q_ARG(QString, "size"), Q_ARG(QVariant, size));
    }
}

void TransferManager::onTransferCompleted(QSharedPointer<TransferItem> transfer) {
    if (transfer.data()->transferType() == Transfers::Download) {
        QMetaObject::invokeMethod(Database::instance(), "removeStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer.data()->id()));
    }

    emit alert(QString("%1 '%3' %4").arg(transfer.data()->transferType() == Transfers::Upload ? tr("Upload of") : tr("Download of")).arg(transfer.data()->title()).arg(tr("completed")));
    this->removeTransfer(transfer);
}

void TransferManager::onTransferCancelled(QSharedPointer<TransferItem> transfer) {
    if (transfer.data()->transferType() == Transfers::Download) {
        QMetaObject::invokeMethod(Database::instance(), "removeStoredDownload", Qt::QueuedConnection, Q_ARG(QString, transfer.data()->id()));
    }

    this->removeTransfer(transfer);
}

void TransferManager::onMaximumConcurrentTransfersChanged(int oldMaximum, int newMaximum) {
    if (newMaximum > oldMaximum) {
        if (newMaximum > this->concurrentTransfers()) {
            if (QSharedPointer<TransferItem> transfer = this->getNextTransfer()) {
                transfer.data()->startTransfer();
            }
        }
    }
    else if (newMaximum < oldMaximum) {
        if (newMaximum < this->concurrentTransfers()) {
            int i = this->transferList()->size() - 1;
            int priority = Transfers::LowPriority;

            while (priority >= Transfers::HighPriority) {
                while (i >= 0) {
                    if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
                        if ((transfer.data()->priority() == priority) && (transfer.data()->status() > Transfers::Queued)) {
                            transfer.data()->setStatus(Transfers::Queued);
                            return;
                        }
                    }

                    i--;
                }

                priority--;
                i = this->transferList()->size() - 1;
            }
        }
    }
}

int TransferManager::concurrentTransfers() const {
    int concurrent = 0;

    for (int i = 0; i < this->transferList()->size(); i++) {
        if (QSharedPointer<TransferItem> transfer = this->transferList()->at(i)) {
            if (transfer.data()->status() > Transfers::Queued) {
                concurrent++;
            }
        }
    }

    return concurrent;
}
