#include "youtubevideodescriptiontab.h"
#include "youtubeplaylistdialog.h"
#include "youtubemetadatadialog.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"
#include "youtubevideolistwindow.h"
#include "youtubechannelwindow.h"
#include "videoplaybackwindow.h"
#include "../base/session.h"
#include "thumbnaillabel.h"
#include <QAction>
#include <QLabel>
#include <QPushButton>
#include <QScrollArea>
#include <QGridLayout>

YouTubeVideoDescriptionTab::YouTubeVideoDescriptionTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_editAction(new QAction(tr("Edit"), this)),
    m_favouritesAction(new QAction(video->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_shareAction(new QAction(tr("Share"), this)),
    m_likeButton(new QPushButton(QIcon(":/images/like.png"), "", this)),
    m_dislikeButton(new QPushButton(QIcon(":/images/dislike.png"), "", this)),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_likesLabel(new QLabel(QString::number(m_video.data()->likes()), this)),
    m_dislikesLabel(new QLabel(QString::number(m_video.data()->dislikes()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Views")).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_editAction);
    this->addAction(m_favouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_watchLaterAction);
    this->addAction(m_queueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);
    this->addAction(m_shareAction);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_likesLabel->setFont(smallFont);
    m_dislikesLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_likeButton->setFixedSize(70, 70);
    m_likeButton->setEnabled((m_session->youtube()->userSignedIn()) && (m_video.data()->artistId() != m_session->youtube()->username()));
    m_dislikeButton->setFixedSize(70, 70);
    m_dislikeButton->setEnabled(m_likeButton->isEnabled());
    m_likesLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);

    grid->addWidget(m_titleLabel, 0, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_likeButton, 2, 0);
    grid->addWidget(m_likesLabel, 2, 1);
    grid->addWidget(m_dislikeButton, 2, 2);
    grid->addWidget(m_dislikesLabel, 2, 3);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 4, Qt::AlignTop);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_likeButton, SIGNAL(clicked()), this, SLOT(likeVideo()));
    this->connect(m_dislikeButton, SIGNAL(clicked()), this, SLOT(dislikeVideo()));
    this->connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    this->connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    this->connect(m_video.data(), SIGNAL(likedChanged(bool)), this, SLOT(onVideoLiked()));
    this->connect(m_video.data(), SIGNAL(dislikedChanged(bool)), this, SLOT(onVideoDisliked()));
    this->connect(m_video.data(), SIGNAL(favouriteChanged(bool)), this, SLOT(onFavouriteChanged(bool)));
    this->connect(m_editAction, SIGNAL(triggered()), this, SLOT(showEditMetadataDialog()));
    this->connect(m_favouritesAction, SIGNAL(triggered()), this, SLOT(toggleFavourite()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(showPlaylistDialog()));
    this->connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideoToWatchLaterPlaylist()));
    this->connect(m_queueAction, SIGNAL(triggered()), this, SLOT(addVideoToPlaybackQueue()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownload()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownload()));
    this->connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));

    this->setMenuActions();
    this->setThumbnails();
    this->onOrientationChanged();
}

void YouTubeVideoDescriptionTab::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void YouTubeVideoDescriptionTab::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

void YouTubeVideoDescriptionTab::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_editAction->setEnabled((signedIn) && (m_video.data()->artistId() == m_session->youtube()->username()));
    m_favouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
}

void YouTubeVideoDescriptionTab::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        this->connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        this->disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void YouTubeVideoDescriptionTab::showEditMetadataDialog() {
    YouTubeMetadataDialog *dialog = new YouTubeMetadataDialog(m_session, m_video, this);
    dialog->open();
}

void YouTubeVideoDescriptionTab::toggleFavourite() {
    if (m_video.data()->favourite()) {
        m_session->youtube()->deleteFromFavourites(QStringList(m_video.data()->favouriteId()));
    }
    else {
        m_session->youtube()->addToFavourites(QStringList(m_video.data()->videoId()));
    }
}

void YouTubeVideoDescriptionTab::onFavouriteChanged(bool favourite) {
    m_favouritesAction->setText(favourite ? tr("Delete from favourites") : tr("Add to favourites"));
}

void YouTubeVideoDescriptionTab::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void YouTubeVideoDescriptionTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();

    this->connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void YouTubeVideoDescriptionTab::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        this->showFacebookDialog();
        break;
    case Services::Twitter:
        this->showTwitterDialog();
        break;
    case Services::Email:
        m_session->sharing()->shareViaEmail(m_video.data()->title(), m_video.data()->url());
        break;
    case Services::Clipboard:
        m_session->sharing()->copyToClipboard(m_video.data()->url());
        break;
    default:
        return;
    }
}

void YouTubeVideoDescriptionTab::showFacebookDialog() {
    if (m_session->sharing()->userSignedInToFacebook()) {
        ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        FacebookAuthDialog *dialog = new FacebookAuthDialog(m_session, this);
        dialog->open();
    }
}

void YouTubeVideoDescriptionTab::showTwitterDialog() {
    if (m_session->sharing()->userSignedInToTwitter()) {
        ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        TwitterAuthDialog *dialog = new TwitterAuthDialog(m_session, this);
        dialog->open();
    }
}

void YouTubeVideoDescriptionTab::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(QStringList(m_video.data()->videoId()), id);
}

void YouTubeVideoDescriptionTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    m_session->youtube()->createPlaylist(playlist, QStringList(m_video.data()->videoId()));
}

void YouTubeVideoDescriptionTab::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(QStringList(m_video.data()->videoId()));
}

void YouTubeVideoDescriptionTab::addVideoToPlaybackQueue() {
    m_session->playbackQueue()->addVideo(m_video);
}

void YouTubeVideoDescriptionTab::addVideoDownload() {
    m_session->addDownloadTransfer(m_video);
}

void YouTubeVideoDescriptionTab::addAudioDownload() {
    m_session->addDownloadTransfer(m_video, true);
}

void YouTubeVideoDescriptionTab::likeVideo() {
    if ((!m_video.data()->liked()) && (!m_video.data()->disliked())) {
        m_session->youtube()->likeVideo(m_video.data()->videoId());
    }
}

void YouTubeVideoDescriptionTab::dislikeVideo() {
    if ((!m_video.data()->liked()) && (!m_video.data()->disliked())) {
        m_session->youtube()->dislikeVideo(m_video.data()->videoId());
    }
}

void YouTubeVideoDescriptionTab::onVideoLiked() {
    m_likesLabel->setText(QString::number(m_video.data()->likes()));
}

void YouTubeVideoDescriptionTab::onVideoDisliked() {
    m_dislikesLabel->setText(QString::number(m_video.data()->dislikes()));
}

void YouTubeVideoDescriptionTab::playVideo() {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_video, m_session->currentWindow());
        window->show();
    }
    else {
        m_session->urlGrabber()->getYouTubeVideoUrl(m_video.data()->videoId());
    }
}

void YouTubeVideoDescriptionTab::showUserChannel() {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(m_session, m_video.data()->artistId(), m_session->currentWindow());
    window->show();
}

void YouTubeVideoDescriptionTab::search(const QString &query) {
    YouTubeVideoListWindow *window = new YouTubeVideoListWindow(m_session, m_session->currentWindow());
    window->search(query, m_session->settings()->searchOrder(), m_session->settings()->searchTime(), m_session->settings()->searchDuration(), m_session->settings()->languageFilter());
    window->show();
}
