#include "youtubevideocommentstab.h"
#include "commentdelegate.h"
#include "youtubecommentdialog.h"
#include "youtubechannelwindow.h"
#include "../base/session.h"
#include "../base/youtubecommentlistmodel.h"
#include "listview.h"
#include <QMenu>
#include <QAction>
#include <QLabel>
#include <QGridLayout>

YouTubeVideoCommentsTab::YouTubeVideoCommentsTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_model(new YouTubeCommentListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_contextMenu(new QMenu(this)),
    m_replyAction(m_contextMenu->addAction(tr("Reply"), this, SLOT(replyToComment()))),
    m_commentAction(new QAction(tr("Add comment"), this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No comments found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_commentAction);

    m_commentAction->setEnabled(m_session->youtube()->userSignedIn());
    m_replyAction->setVisible(m_session->youtube()->userSignedIn());
    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    CommentDelegate *delegate = new CommentDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_listView->setUniformItemSizes(false);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreComments()));
    this->connect(delegate, SIGNAL(showUserChannel(QModelIndex)), this, SLOT(showUserChannel(QModelIndex)));
    this->connect(delegate, SIGNAL(getInitialComment(int)), m_model, SLOT(getInitialComment(int)));
    this->connect(m_commentAction, SIGNAL(triggered()), this, SLOT(showCommentDialog()));

    if (m_session->youtube()->userSignedIn()) {
        this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    }

    m_model->getComments(m_video.data()->videoId());

    this->onOrientationChanged();
}

void YouTubeVideoCommentsTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeVideoCommentsTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeVideoCommentsTab::onLoadingChanged(bool loading) {
    m_session->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void YouTubeVideoCommentsTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void YouTubeVideoCommentsTab::showCommentDialog(QSharedPointer<CommentItem> initialComment) {
    YouTubeCommentDialog *dialog = new YouTubeCommentDialog(m_session, m_video, initialComment, this);
    dialog->open();
}

void YouTubeVideoCommentsTab::replyToComment() {
    this->showCommentDialog(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoCommentsTab::showUserChannel(const QModelIndex &index) {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(m_session, index.data(AbstractCommentListModel::ArtistIdRole).toString(), m_session->currentWindow());
    window->show();
}

void YouTubeVideoCommentsTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_replyAction);
}
