#include "youtubesubscriptionlistmodel.h"
#include "session.h"
#include "thumbnailcache.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

YouTubeSubscriptionListModel::YouTubeSubscriptionListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache())
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[SubscriptionIdRole] = "subscriptionId";
    roles[UsernameRole] = "username";
    roles[ChannelNameRole] = "channelName";
    roles[AvatarRole] = "avatar";
    roles[AvatarUrlRole] = "avatarUrl";
    roles[VideoCountRole] = "videoCount";
    roles[DescriptionRole] = "description";
    roles[ViewCountRole] = "viewCount";
    roles[SubscriberCountRole] = "subscriberCount";
    roles[SubscriptionRole] = "subscription";
    this->setRoleNames(roles);

    if (this->session()) {
        if (!this->session()->youtube()->subscriptionsLoaded()) {
            this->session()->youtube()->getSubscriptions();
        }

        this->connect(this->session()->youtube(), SIGNAL(subscriptionAddedToCache(int)), this, SLOT(onSubscriptionAdded(int)));
        this->connect(this->session()->youtube(), SIGNAL(subscriptionRemovedFromCache(int)), this, SLOT(onSubscriptionRemoved(int)));
    }
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

YouTubeSubscriptionListModel::~YouTubeSubscriptionListModel() {
    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void YouTubeSubscriptionListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        if (!this->session()->youtube()->subscriptionsLoaded()) {
            this->session()->youtube()->getSubscriptions();
        }

        this->connect(this->session()->youtube(), SIGNAL(subscriptionAddedToCache(int)), this, SLOT(onSubscriptionAdded(int)));
        this->connect(this->session()->youtube(), SIGNAL(subscriptionRemovedFromCache(int)), this, SLOT(onSubscriptionRemoved(int)));
    }
}

void YouTubeSubscriptionListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void YouTubeSubscriptionListModel::onSubscriptionAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

void YouTubeSubscriptionListModel::onSubscriptionRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

int YouTubeSubscriptionListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->youtube()->subscriptions()->size();
}

QVariant YouTubeSubscriptionListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->id();
    case SubscriptionIdRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->subscriptionId();
    case UsernameRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->username();
    case ChannelNameRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->channelName();
#ifndef QML_USER_INTERFACE
    case AvatarRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->youtube()->subscriptions()->at(index.row()).data()->avatarUrl(), QSize(60, 60));
#endif
    case AvatarUrlRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->avatarUrl();
    case VideoCountRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->videoCount();
    case SubscriberCountRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->subscriberCount();
    case SubscriptionRole:
        return this->session()->youtube()->subscriptions()->at(index.row()).data()->subscription();
    default:
        return QVariant();
    }
}

QVariant YouTubeSubscriptionListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<UserItem> YouTubeSubscriptionListModel::get(int row) const {
    return QSharedPointer<UserItem>(this->session()->youtube()->subscriptions()->at(row));
}

UserItem* YouTubeSubscriptionListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    UserItem *user = new UserItem(this->session()->youtube()->subscriptions()->at(row).data());
    this->connect(this->session()->youtube(), SIGNAL(subscriptionChanged(QString,bool,QString)), user, SLOT(onSubscriptionChanged(QString,bool,QString)));
    QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);

    return user;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void YouTubeSubscriptionListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
