#include "vimeoprofiletab.h"
#include "vimeovideolistwindow.h"
#include "../base/session.h"
#include "separatorlabel.h"
#include "avatarlabel.h"
#include <QLabel>
#include <QPushButton>
#include <QScrollArea>
#include <QAction>
#include <QGridLayout>
#include <QDesktopServices>

VimeoProfileTab::VimeoProfileTab(Session *session, const QString &userId, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_user(0),
    m_avatar(new AvatarLabel(this)),
    m_channelLabel(new QLabel(this)),
    m_videosLabel(new QLabel(this)),
    m_locationLabel(new QLabel(this)),
    m_websiteLabel(new QLabel(this)),
    m_descriptionLabel(new QLabel(this)),
    m_noProfileLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No profile found")), this)),
    m_subscriptionButton(new QPushButton(tr("Subscribe"))),
    m_watchLaterAction(new QAction(tr("Show 'Watch later' album"), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(m_channelLabel, 0, 0, 1, 3);
    grid->addWidget(m_avatar, 1, 0, 1, 1, Qt::AlignTop);
    grid->addWidget(m_videosLabel, 1, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_subscriptionButton, 2, 2);
    grid->addWidget(new SeparatorLabel(tr("Profile"), this), 3, 0, 1, 3);
    grid->addWidget(m_locationLabel, 4, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_websiteLabel, 5, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(new SeparatorLabel(tr("Description"), this), 6, 0, 1, 3);
    grid->addWidget(m_descriptionLabel, 7, 0, 1, 3);

    m_grid->setContentsMargins(0, 0 ,0, 0);
    m_grid->addWidget(m_scrollArea);
    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_descriptionLabel->setWordWrap(true);
    m_scrollArea->hide();
    m_noProfileLabel->hide();
    m_videosLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_subscriptionButton->setFixedWidth(150);
    m_subscriptionButton->setVisible(m_session->vimeo()->userSignedIn());

    this->connect(m_session->vimeo(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    this->connect(m_session->vimeo(), SIGNAL(error(QString)), this, SLOT(onProfileError()));
    this->connect(m_websiteLabel, SIGNAL(linkActivated(QString)), this, SLOT(showUserWebsite()));
    this->connect(m_subscriptionButton, SIGNAL(clicked()), this, SLOT(toggleSubscription()));

    if ((userId == "me") || (userId == m_session->vimeo()->username())) {
        m_session->vimeo()->getCurrentUserProfile();
        m_subscriptionButton->hide();
        this->addAction(m_watchLaterAction);
        this->connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(showWatchLaterAlbum()));
    }
    else {
        m_session->vimeo()->getUserProfile(userId);
    }

    m_session->toggleBusy(true);
    this->onOrientationChanged();
}

void VimeoProfileTab::setLandscapeLayout() {
    m_noProfileLabel->setFixedSize(800, 400);
}

void VimeoProfileTab::setPortraitLayout() {
    m_noProfileLabel->setFixedSize(480, 700);
}

void VimeoProfileTab::setUser(QSharedPointer<UserItem> user) {
    m_user = user;
    m_avatar->setName(m_user.data()->username());
    m_channelLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(m_user.data()->channelName()));
    m_videosLabel->setText(!m_user.data()->videoCount() ? tr("No videos") : QString ("%1 %2").arg(m_user.data()->videoCount()).arg(tr("videos")));
    m_locationLabel->setText(QString("%1: %2").arg(tr("Location")).arg(m_user.data()->location().isEmpty() ? tr("Unknown") : m_user.data()->location()));
    m_websiteLabel->setText(m_user.data()->websiteUrl().isEmpty() ? tr("No website") : QString("%1: <a href='%2'>%2</a>").arg("Website").arg(m_user.data()->websiteUrl().toString()));
    m_descriptionLabel->setText(m_user.data()->description().isEmpty() ? tr("No description") : m_user.data()->description());
    m_subscriptionButton->setText(m_user.data()->subscribed() ? tr("Unsubscribe") : tr("Subscribe"));

    m_scrollArea->show();

    this->setAvatar();
    m_session->toggleBusy(false);

    this->connect(m_user.data(), SIGNAL(subscribedChanged(bool)), this, SLOT(onSubscribedChanged(bool)));
    this->disconnect(m_session->vimeo(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    this->disconnect(m_session->vimeo(), SIGNAL(error(QString)), this, SLOT(onProfileError()));
}

void VimeoProfileTab::setAvatar() {
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_user.data()->avatarUrl(), QSize(128, 128));

    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
        this->disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
    else {
        this->connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
}

void VimeoProfileTab::onProfileError() {
    m_noProfileLabel->show();
}

void VimeoProfileTab::toggleSubscription() {
    if (m_user.data()->subscribed()) {
        m_session->vimeo()->unsubscribe(m_user.data()->id());
    }
    else {
        m_session->vimeo()->subscribe(m_user.data()->id());
    }
}

void VimeoProfileTab::onSubscribedChanged(bool subscribed) {
    m_subscriptionButton->setText(subscribed ? tr("Unsubscribe") : tr("Subscribe"));
}

void VimeoProfileTab::showUserWebsite() {
    QDesktopServices::openUrl(m_user.data()->websiteUrl());
}

void VimeoProfileTab::showWatchLaterAlbum() {
    VimeoVideoListWindow *window = new VimeoVideoListWindow(m_session, m_session->currentWindow());
    window->setWindowTitle(tr("Watch later"));
    window->setVideoFeed(VIMEO_WATCH_LATER_FEED);
    window->show();
}
