#include "dailymotionrelatedvideostab.h"
#include "dailymotionplaylistdialog.h"
#include "videolistdelegate.h"
#include "dailymotionvideoinfowindow.h"
#include "dailymotionvideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/session.h"
#include "../base/dailymotionvideolistmodel.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>

DailymotionRelatedVideosTab::DailymotionRelatedVideosTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_model(new DailymotionVideoListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_contextMenu(new QMenu(this)),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    this->addAction(m_addToFavouritesAction);
    this->addAction(m_addToPlaylistAction);
    this->addAction(m_addToQueueAction);
    this->addAction(m_videoDownloadAction);
    this->addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    this->connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    this->connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    this->connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    this->connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    m_model->getRelatedVideos(m_video.data()->videoId());

    this->setMenuActions();
    this->onOrientationChanged();
}

void DailymotionRelatedVideosTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void DailymotionRelatedVideosTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(400, 700);
}

void DailymotionRelatedVideosTab::setMenuActions() {
    bool signedIn = m_session->dailymotion()->userSignedIn();

    m_addToFavouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_addToFavouritesContextAction->setVisible(signedIn);
    m_addToPlaylistContextAction->setVisible(signedIn);
}

void DailymotionRelatedVideosTab::onLoadingChanged(bool loading) {
    m_session->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void DailymotionRelatedVideosTab::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void DailymotionRelatedVideosTab::addVideosToFavourites() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::addVideosToPlaylist() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::addVideoDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addVideoDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::addAudioDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addAudioDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::queueVideos() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Add to queue"), SLOT(queueVideos()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionRelatedVideosTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToFavouritesContextAction);
}

void DailymotionRelatedVideosTab::addVideoToFavourites() {
    m_session->dailymotion()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionRelatedVideosTab::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void DailymotionRelatedVideosTab::addVideoToPlaylist(const QString &id) {
    m_session->dailymotion()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void DailymotionRelatedVideosTab::addVideoToNewPlaylist(const QVariantMap &playlist) {
    m_session->dailymotion()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionRelatedVideosTab::addVideoDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionRelatedVideosTab::addAudioDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void DailymotionRelatedVideosTab::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionRelatedVideosTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void DailymotionRelatedVideosTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(m_session, video, m_session->currentWindow());
    window->show();
}

void DailymotionRelatedVideosTab::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), m_session->currentWindow());
        window->show();
    }
    else {
        m_session->urlGrabber()->getDailymotionVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
