#include "vimeouseralbumlistmodel.h"
#include "session.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

VimeoUserAlbumListModel::VimeoUserAlbumListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache())
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[AlbumIdRole] = "albumId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[VideoCountRole] = "videoCount";
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[DescriptionRole] = "description";
    this->setRoleNames(roles);

    if (this->session()) {
        if (!this->session()->vimeo()->albumsLoaded()) {
            this->session()->vimeo()->getAlbums();
        }

        this->connect(this->session()->vimeo(), SIGNAL(albumAddedToCache(int)), this, SLOT(onAlbumAdded(int)));
        this->connect(this->session()->vimeo(), SIGNAL(albumRemovedFromCache(int)), this, SLOT(onAlbumRemoved(int)));
        this->connect(this->session()->vimeo(), SIGNAL(albumUpdated(int)), this, SLOT(onAlbumUpdated(int)));
    }
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void VimeoUserAlbumListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        if (!this->session()->vimeo()->albumsLoaded()) {
            this->session()->vimeo()->getAlbums();
        }

        this->connect(this->session()->vimeo(), SIGNAL(albumAddedToCache(int)), this, SLOT(onAlbumAdded(int)));
        this->connect(this->session()->vimeo(), SIGNAL(albumRemovedFromCache(int)), this, SLOT(onAlbumRemoved(int)));
        this->connect(this->session()->vimeo(), SIGNAL(albumUpdated(int)), this, SLOT(onAlbumUpdated(int)));
    }
}

void VimeoUserAlbumListModel::onAlbumAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

void VimeoUserAlbumListModel::onAlbumRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

void VimeoUserAlbumListModel::onAlbumUpdated(int row) {
    emit dataChanged(this->index(row), this->index(row));
}

int VimeoUserAlbumListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->vimeo()->albums()->size();
}

QVariant VimeoUserAlbumListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->id();
    case AlbumIdRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->playlistId();
    case TitleRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->title();
    case ArtistRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->artist();
    case DateRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->date();
    case VideoCountRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->videoCount();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->vimeo()->albums()->at(index.row()).data()->thumbnailUrl(), QSize(120, 90));
#endif
    case ThumbnailUrlRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->thumbnailUrl();
    case DescriptionRole:
        return this->session()->vimeo()->albums()->at(index.row()).data()->description();
    default:
        return QVariant();
    }
}

QVariant VimeoUserAlbumListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

QSharedPointer<PlaylistItem> VimeoUserAlbumListModel::get(int row) const {
    return QSharedPointer<PlaylistItem>(this->session()->vimeo()->albums()->at(row));
}

PlaylistItem* VimeoUserAlbumListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    PlaylistItem* album = new PlaylistItem(this->session()->vimeo()->albums()->at(row).data());
    QDeclarativeEngine::setObjectOwnership(album, QDeclarativeEngine::JavaScriptOwnership);

    return album;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void VimeoUserAlbumListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
