#include "urlgrabber.h"
#include "cookiejar.h"
#include "json.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QDebug>

using namespace QtJson;

UrlGrabber::UrlGrabber(UrlGrabber::Mode mode, QObject *parent) :
    QObject(parent),
    m_mode(mode),
    m_nam(0),
    m_dailymotionPlaybackNam(0),
    m_busy(false)
{
    m_youtubeList << 37 << 22 << 35 << 34 << 18;
    m_dailymotionList << "720" << "480" << "380";

    m_youtubeMap[37] = "1080P AVC1";
    m_youtubeMap[22] = "720P AVC1";
    m_youtubeMap[35] = "480P FLV";
    m_youtubeMap[34] = "360P FLV";
    m_youtubeMap[18] = QString("%1 AVC1").arg(tr("Normal (upto 360P)"));

    m_dailymotionMap["720"] = "720P AVC1";
    m_dailymotionMap["480"] = "480P AVC1";
    m_dailymotionMap["380"] = QString("%1 AVC1").arg(tr("Normal"));

    m_vimeoMap["mobile"] = tr("Mobile");
    m_vimeoMap["sd"] = tr("Normal");
    m_vimeoMap["hd"] = "720P AVC1";

    if (m_mode == UrlGrabber::PlaybackMode) {
        m_dailymotionPlaybackNam = new QNetworkAccessManager(this);
        m_dailymotionPlaybackNam->setCookieJar(new CookieJar(CookieJar::PlaybackMode, m_dailymotionPlaybackNam));
    }
}

void UrlGrabber::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            emit busy(message, numberOfOperations);
        }
        else {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void UrlGrabber::getVideoUrl(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getYouTubeVideoUrl(id);
        break;
    case Services::Dailymotion:
        this->getDailymotionVideoUrl(id);
        break;
    case Services::Vimeo:
        this->getVimeoVideoUrl(id);
        break;
    default:
        return;
    }
}

void UrlGrabber::getYouTubeVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url(QString("http://www.youtube.com/get_video_info?&video_id=%1&el=detailpage&ps=default&eurl=&gl=US&hl=en").arg(id));
    QNetworkRequest request;
    request.setUrl(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseYouTubeVideoPage()));
}

void UrlGrabber::parseYouTubeVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QMap<int, QString> formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        response = QByteArray::fromPercentEncoding(response.toUtf8()).replace("%2C", ",").replace("sig=", "signature=").replace(",signature=", "&signature=");
        QStringList urlStrings = response.split("url=", QString::SkipEmptyParts);
        QStringList sigs = response.split("signature=");
        QStringList signatures;

        for (int i = 1; i < sigs.size(); i++) {
            signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
        }

        foreach (QString urlString, urlStrings) {
            if (urlString.startsWith("http")) {
                QUrl url(urlString.section(QRegExp("(,|&|)itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();

                if (key > 0) {
                    if (!signatures.isEmpty()) {
                        url.removeQueryItem("signature");
                        url.addQueryItem("signature", signatures.takeFirst());
                    }

                    formats[key] = url.toString();
                }
            }
        }

        QString videoUrl;
        int index = 0;

        while ((videoUrl.isEmpty()) && (index < m_youtubeList.size())) {
            int format = m_youtubeList.at(index);

            if (m_youtubeSet.contains(format)) {
                videoUrl = formats.value(format, "");
            }

            index++;

        }

        if (!videoUrl.startsWith("http")) {
            emit error(tr("Unable to retrieve video. Access may be restricted"));
        }
        else {
            emit gotVideoUrl(QUrl(videoUrl));
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://www.dailymotion.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = this->networkAccessManager()->get(request);
    }

    this->connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));
}

void UrlGrabber::parseDailymotionVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(QByteArray::fromPercentEncoding(reply->readAll()).replace("\\", ""));
    QString manifestUrl = response.section("autoURL\":\"", 1, 1).section('"', 0, 0);

    if (manifestUrl.startsWith("http://")) {
        this->getDailymotionFormats(QUrl(manifestUrl));
    }
    else {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionFormats(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = this->networkAccessManager()->get(request);
    }

    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionFormats()));
}

void UrlGrabber::checkDailymotionFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QVariantMap formats = Json::parse(response).toMap();
    QVariantList formatList = formats.value("alternates").toList();
    QString manifestUrl;
    int i = 0;
    int ii = 0;

    while ((manifestUrl.isEmpty()) && (i < m_dailymotionList.size())) {
        while ((manifestUrl.isEmpty()) && (ii < formatList.size())) {
            QVariantMap map = formatList.at(ii).toMap();
            QByteArray name = map.value("name").toByteArray();

            if ((name == m_dailymotionList.at(i)) && (m_dailymotionSet.contains(name))) {
                manifestUrl = map.value("template").toString();
            }

            ii++;
        }

        ii = 0;
        i++;
    }

    if (manifestUrl.startsWith("http://")) {
        this->getDailymotionUrl(QUrl(manifestUrl));
    }
    else {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionUrl(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = this->networkAccessManager()->get(request);
    }

    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionUrl()));
}

void UrlGrabber::checkDailymotionUrl() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QVariantMap map = Json::parse(response).toMap();
    QString urlTemplate = map.value("template").toString().remove("frag($fragment$)/");

    if (urlTemplate.isEmpty()) {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QString host = reply->request().url().host();
        QUrl videoUrl(QString("http://%1%2.mp4").arg(host).arg(urlTemplate.section('.', 0, -2)));
        emit gotVideoUrl(videoUrl);
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://vimeo.com/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoVideoPage()));
}

void UrlGrabber::parseVimeoVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    bool ok;
    QVariantMap paramMap = Json::parse(params, ok).toMap();

    if (!ok) {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");
        QString quality;

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QVariantList qualities = formatMap.value(codec).toList();

            int i = 0;

            while ((quality.isEmpty()) && (i < qualities.size())) {
                QByteArray q = qualities.at(i).toByteArray();

                if (m_vimeoSet.contains(q)) {
                    quality = q;
                }

                i++;
            }
        }

        if (quality.isEmpty()) {
            quality = "mobile";
        }

        QString timeStamp = requestMap.value("timestamp").toString();
        QString signature = requestMap.value("signature").toString();
        QString id = videoMap.value("id").toString();
        QUrl url;

        if (quality == "mobile") {
            url.setUrl(QString("http://player.vimeo.com/play_redirect?quality=mobile&clip_id=%1&time=%2&sig=%3&type=mobile_site").arg(id).arg(timeStamp).arg(signature));
        }
        else {
            url.setUrl(QString("http://player.vimeo.com/play_redirect?quality=%1&codecs=%2&clip_id=%3&time=%4&sig=%5&type=html5_desktop_local").arg(quality).arg(codec).arg(id).arg(timeStamp).arg(signature));
        }

        this->getVimeoRedirect(url);
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoRedirect()));
}

void UrlGrabber::checkVimeoRedirect() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isEmpty()) {
        qDebug() << reply->request().url();
        emit error(tr("Unable to retrieve video. Note that some videos are not available in 'Mobile' quality"));
    }
    else {
        emit gotVideoUrl(redirect);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVideoFormats(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getAvailableYouTubeVideoFormats(id);
        break;
    case Services::Dailymotion:
        this->getAvailableDailymotionVideoFormats(id);
        break;
    case Services::Vimeo:
        this->getAvailableVimeoVideoFormats(id);
        break;
    default:
        return;
    }
}

void UrlGrabber::getAvailableYouTubeVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url(QString("http://www.youtube.com/get_video_info?&video_id=%1&el=detailpage&ps=default&eurl=&gl=US&hl=en").arg(id));
    QNetworkRequest request;
    request.setUrl(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkYouTubeVideoFormats()));
}

void UrlGrabber::checkYouTubeVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QVariantList formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video formats. Access may be restricted"));
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        response = QByteArray::fromPercentEncoding(response.toUtf8()).replace("%2C", ",").replace("sig=", "signature=").replace(",signature=", "&signature=");
        QStringList urlStrings = response.split("url=", QString::SkipEmptyParts);
        QStringList sigs = response.split("signature=");
        QStringList signatures;

        for (int i = 1; i < sigs.size(); i++) {
            signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
        }

        foreach (QString urlString, urlStrings) {
            if (urlString.startsWith("http")) {
                QUrl url(urlString.section(QRegExp("(,|&|)itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();

                if (key > 0) {
                    if (!signatures.isEmpty()) {
                        if ((key == 22) || (key == 18)) {
                            url.removeQueryItem("signature");
                            url.addQueryItem("signature", signatures.takeFirst());

                            QVariantMap format;
                            format["name"] = m_youtubeMap[key];
                            format["service"] = Services::YouTube;
                            format["url"] = url;
                            formats.append(format);
                        }
                        else {
                            signatures.takeFirst();
                        }
                    }
                }
            }
        }
    }

    emit gotVideoFormats(formats);

    reply->deleteLater();
}

void UrlGrabber::getAvailableDailymotionVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://www.dailymotion.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = this->networkAccessManager()->get(request);
    }

    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionVideoFormats()));
}

void UrlGrabber::checkDailymotionVideoFormats() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(QByteArray::fromPercentEncoding(reply->readAll()).replace("\\", ""));
    QString manifestUrl = response.section("autoURL\":\"", 1, 1).section('"', 0, 0);

    if (manifestUrl.startsWith("http://")) {
        this->getDailymotionVideoQualityFormats(QUrl(manifestUrl));
    }
    else {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video formats. Access may be restricted"));
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionVideoQualityFormats(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply;

    if (m_mode == UrlGrabber::PlaybackMode) {
        reply = m_dailymotionPlaybackNam->get(request);
    }
    else {
        reply = this->networkAccessManager()->get(request);
    }

    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionVideoQualityFormats()));
}

void UrlGrabber::checkDailymotionVideoQualityFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QVariantList formats;
    QString response(reply->readAll());
    QVariantMap formatMap = Json::parse(response).toMap();
    QVariantList formatList = formatMap.value("alternates").toList();
    QString manifestUrl;
    int i = 0;
    int ii = 0;

    while ((manifestUrl.isEmpty()) && (i < m_dailymotionList.size())) {
        while ((manifestUrl.isEmpty()) && (ii < formatList.size())) {
            QVariantMap map = formatList.at(ii).toMap();
            QString name = map.value("name").toString();

            if (name == m_dailymotionList.at(i)) {
                QVariantMap format;
                format["name"] = m_dailymotionMap[name];
                format["service"] = Services::Dailymotion;
                format["url"] = map.value("template").toUrl();
                formats.append(format);
            }

            ii++;
        }

        ii = 0;
        i++;
    }

    emit gotVideoFormats(formats);

    reply->deleteLater();
}

void UrlGrabber::getAvailableVimeoVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://vimeo.com/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoFormats()));
}

void UrlGrabber::checkVimeoVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    bool ok;
    QVariantMap paramMap = Json::parse(params, ok).toMap();

    if (!ok) {
        qDebug() << reply->request().url();
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");
        QString quality;

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QVariantList qualities = formatMap.value(codec).toList();
            QVariantList formats;

            int i = 0;

            while ((quality.isEmpty()) && (i < qualities.size())) {
                QByteArray q = qualities.at(i).toByteArray();
                QString timeStamp = requestMap.value("timestamp").toString();
                QString signature = requestMap.value("signature").toString();
                QString id = videoMap.value("id").toString();

                QVariantMap format;
                format["name"] = m_vimeoMap[q];
                format["service"] = Services::Vimeo;

                if (q == "mobile") {
                    format["url"] = QUrl(QString("http://player.vimeo.com/play_redirect?quality=mobile&clip_id=%1&time=%2&sig=%3&type=mobile_site").arg(id).arg(timeStamp).arg(signature));
                }
                else {
                    format["url"] = QUrl(QString("http://player.vimeo.com/play_redirect?quality=%1&codecs=%2&clip_id=%3&time=%4&sig=%5&type=html5_desktop_local").arg(quality).arg(codec).arg(id).arg(timeStamp).arg(signature));
                }

                formats.append(format);

                i++;
            }

            emit gotVideoFormats(formats);
        }
        else {
            qDebug() << reply->request().url();
            qDebug() << response;
            emit error(tr("Unable to retrieve video formats. Access may be restricted"));
        }
    }

    reply->deleteLater();
}
