#include "socialvideolistmodel.h"
#include "session.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

SocialVideoListModel::SocialVideoListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session),
    m_cache(!session ? 0 : session->newThumbnailCache()),
    m_thumbnailPressedRow(-1)
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[VideoIdRole] = "videoId";
    roles[FavouriteIdRole] = "favouriteId";
    roles[PlaylistVideoIdRole] = "playlistVideoId";
    roles[TitleRole] = "title";
    roles[ArtistRole] = "artist";
    roles[DateRole] = "date";
    roles[DurationRole] = "duration";
    roles[ViewCountRole] = "viewCount";
    roles[RatingCountRole] = "ratingCount";
    roles[RatingRole] = "rating";
    roles[ThumbnailRole] = "thumbnail";
    roles[ThumbnailUrlRole] = "thumbnailUrl";
    roles[ArchiveRole] = "archive";
    roles[ServiceRole] = "service";
    roles[FavouriteRole] = "favourite";
    roles[SelectedRole] = "selected";
    roles[ThumbnailPressedRole] = "thumbnailPressed";
    this->setRoleNames(roles);

    if (this->session()) {
        this->session()->sharing()->getVideos();
        this->connect(this->session()->sharing(), SIGNAL(videoAddedToCache(int)), this, SLOT(onVideoAdded(int)));
    }
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

SocialVideoListModel::~SocialVideoListModel() {
    if (this->thumbnailCache()) {
        delete this->thumbnailCache();
    }
}

void SocialVideoListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        this->session()->sharing()->getVideos();
        this->connect(this->session()->sharing(), SIGNAL(videoAddedToCache(int)), this, SLOT(onVideoAdded(int)));
    }
}

void SocialVideoListModel::setThumbnailCache(ThumbnailCache *cache) {
    m_cache = cache;
#ifndef QML_USER_INTERFACE
    if (this->thumbnailCache()) {
        this->connect(this->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
    }
#endif
}

void SocialVideoListModel::onVideoAdded(int row) {
    this->beginInsertRows(QModelIndex(), row, row);
    this->endInsertRows();
    emit countChanged(this->rowCount());
}

int SocialVideoListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return this->session()->sharing()->videos()->size();
}

QVariant SocialVideoListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->id();
    case VideoIdRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->videoId();
    case TitleRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->title();
    case ArtistRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->artist();
    case DateRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->date();
    case DurationRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->duration();
    case ViewCountRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->viewCount();
    case RatingCountRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->ratingCount();
    case RatingRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->rating();
#ifndef QML_USER_INTERFACE
    case ThumbnailRole:
        return !this->thumbnailCache() ? QVariant() : this->thumbnailCache()->thumbnail(this->session()->sharing()->videos()->at(index.row()).data()->thumbnailUrl(), QSize(160, 120));
#endif
    case ThumbnailUrlRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->thumbnailUrl();
    case ServiceRole:
        return this->session()->sharing()->videos()->at(index.row()).data()->service();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

QSharedPointer<VideoItem> SocialVideoListModel::get(int row) const {
    return QSharedPointer<VideoItem>(this->session()->sharing()->videos()->at(row));
}

VideoItem* SocialVideoListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    VideoItem *video = new VideoItem(this->session()->sharing()->videos()->at(row).data());

    if (video->service() == Services::YouTube) {
        this->connect(this->session()->youtube(), SIGNAL(favouriteChanged(QString,bool,QString)), video, SLOT(onFavouriteChanged(QString,bool,QString)));
        this->connect(this->session()->youtube(), SIGNAL(videoLiked(QString)), video, SLOT(onVideoLiked(QString)));
        this->connect(this->session()->youtube(), SIGNAL(videoDisliked(QString)), video, SLOT(onVideoDisliked(QString)));
    }
    else if (video->service() == Services::Dailymotion) {
        this->connect(this->session()->dailymotion(), SIGNAL(favouriteChanged(QString,bool)), video, SLOT(onFavouriteChanged(QString,bool)));
    }

    QDeclarativeEngine::setObjectOwnership(video, QDeclarativeEngine::JavaScriptOwnership);

    return video;
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void SocialVideoListModel::toggleSelected(int row) {
    if (!m_selectedRows.contains(row)) {
        m_selectedRows.append(row);
    }
    else {
        m_selectedRows.removeOne(row);
    }

    emit dataChanged(this->index(row), this->index(row));
}

void SocialVideoListModel::selectAll() {
    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}

void SocialVideoListModel::selectNone() {
    m_selectedRows.clear();

    if (this->rowCount() > 0) {
        emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    }
}

QList< QSharedPointer<VideoItem> > SocialVideoListModel::selectedItems() const {
    QList< QSharedPointer<VideoItem> > checked;

    for (int i = 0; i < m_selectedRows.size(); i++) {
        checked.append(this->get(m_selectedRows.at(i)));
    }

    return checked;
}

QList<VideoItem*> SocialVideoListModel::selectedItemsFromQML() const {
    QList<VideoItem*> items;
#ifdef QML_USER_INTERFACE
    for (int i = 0; i < m_selectedRows.size(); i++) {
        items.append(this->getFromQML(m_selectedRows.at(i)));
    }
#endif
    return items;
}

void SocialVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    this->session()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void SocialVideoListModel::queueSelectedVideos() {
    this->session()->playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}

void SocialVideoListModel::onThumbnailReady() {
#ifndef QML_USER_INTERFACE
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
#endif
}
