#include "settings.h"
#include <qplatformdefs.h>
#include <QDir>
#include <QDesktopServices>
#include <QNetworkProxy>
#include <QAuthenticator>
#ifdef Q_OS_SYMBIAN
#include <QSystemDeviceInfo>
#endif

Settings::Settings(QObject *parent) :
    QSettings(QString("cuteTube"), QString("cuteTube"), parent),
    m_autoTransfers(true),
    m_maxConcurrentTransfers(1),
    m_safeSearch(false),
    m_searchSite(Services::YouTube),
    m_showSearchSite(true),
    m_searchTime(Queries::AllTime),
    m_showSearchTime(false),
    m_searchDuration(Queries::Any),
    m_showSearchDuration(false),
    m_searchType(Queries::Videos),
    m_showSearchType(true),
    m_languageFilter("all"),
    m_showSearchLanguage(false),
    m_theme("dark"),
    m_activeColor("#0881cb"),
    m_activeColorString("color7"),
    m_language("en"),
    m_mediaplayer("cutetube"),
    m_backgroundPlaybackEnabled(false),
    m_orientation(ScreenOrientation::Automatic),
    m_monitorClipboard(false),
    m_proxyType(NetworkProxyType::HttpProxy),
    m_proxyPort(80),
    m_eventFeedEnabled(false),
    m_eventFeedInstalled(false),
    m_cuteTubePlayerEnabled(true)
{
}

Settings::~Settings() {
    this->saveSettings();
}

void Settings::restoreSettings() {
    this->beginGroup("Media");
    this->setYouTubePlaybackQualitySet(this->value("youtubePlaybackQuality", QVariantList()).toList());
    this->setYouTubeDownloadQualitySet(this->value("youtubeDownloadQuality", QVariantList()).toList());
    this->setDailymotionPlaybackQualitySet(this->value("dailymotionPlaybackQuality", QVariantList()).toList());
    this->setDailymotionDownloadQualitySet(this->value("dailymotionDownloadQuality", QVariantList()).toList());
#ifdef Q_WS_MAEMO_5
    this->setVimeoPlaybackQualitySet(this->value("vimeoPlaybackQuality", QVariantList()).toList());
    this->setVimeoDownloadQualitySet(this->value("vimeoDownloadQuality", QVariantList()).toList());
#else
    this->setVimeoPlaybackQualitySet(this->value("vimeoPlaybackQuality", QVariantList() << "sd").toList());
    this->setVimeoDownloadQualitySet(this->value("vimeoDownloadQuality", QVariantList() << "sd").toList());
#endif
    this->setUploadAttribute(this->value("uploadAttribute", tr("Uploaded via cuteTube")).toString());
#ifdef Q_OS_SYMBIAN
    this->setCuteTubePlayerEnabled();
    this->setMediaPlayer(this->cuteTubePlayerEnabled() ? this->value("mediaPlayer", QString("cutetube")).toString() : QString("mediaplayer"));
#else
    this->setMediaPlayer(this->value("mediaPlayer", QString("cutetube")).toString());
    this->setMediaPlayerCommand(this->value("mediaPlayerCommand", QString()).toString());
#endif
#ifdef MEEGO_EDITION_HARMATTAN
    this->setBackgroundPlaybackEnabled(this->value("backgroundPlaybackEnabled", false).toBool());
#endif
    this->setLanguageFilter(this->value("languageFilter", QString("all")).toString());
    this->endGroup();

    this->beginGroup("Transfers");
    this->setStartTransfersAutomatically(this->value("startTransfersAutomatically", true).toBool());
    this->setMaximumConcurrentTransfers(this->value("maximumConcurrentTransfers", 1).toInt());
#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->setDownloadPath(this->value("downloadPath", QString("/home/user/MyDocs/cuteTube/")).toString());
#elif defined (Q_OS_SYMBIAN)
    this->setDownloadPath(this->value("downloadPath", QString("E:/cuteTube/")).toString());
#else
    this->setDownloadPath(this->value("downloadPath", QDesktopServices::storageLocation(QDesktopServices::HomeLocation).append("/cuteTube/")).toString());
#endif
    this->endGroup();

    this->beginGroup("Search");
    this->setSafeSearch(this->value("safeSearch", false).toBool());
    this->setSearchSite(static_cast<Services::VideoService>(this->value("searchSite", 0).toInt()));
    this->setShowSearchSite(this->value("showSearchSite", true).toBool());
    this->setSearchOrder(static_cast<Queries::QueryOrder>(this->value("searchOrder", 0).toInt()));
    this->setShowSearchOrder(this->value("showSearchOrder", true).toBool());
    this->setSearchTime(static_cast<Queries::TimeFilter>(this->value("searchTime", 0).toInt()));
    this->setShowSearchTime(this->value("showSearchTime", false).toBool());
    this->setSearchDuration(static_cast<Queries::DurationFilter>(this->value("searchDuration", 0).toInt()));
    this->setShowSearchDuration(this->value("showSearchDuration", false).toBool());
    this->setSearchType(static_cast<Queries::QueryType>(this->value("searchType", 0).toInt()));
    this->setShowSearchType(this->value("showSearchType", true).toBool());
    this->setShowSearchLanguage(this->value("showSearchLanguage", true).toBool());
    this->endGroup();

#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->beginGroup("Events");
    this->setEventFeedEnabled((this->value("eventFeedEnabled", false).toBool()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

    this->beginGroup("Appearance");
    this->setTheme(this->value("theme", QString("dark")).toString());
    this->setActiveColor(this->value("activeColor", QString("#0881cb")).toString());
    this->setActiveColorString(this->value("activeColorString", QString("color7")).toString());
    this->endGroup();

    this->beginGroup("System");
    this->setLanguage(this->value("language", QString("en")).toString());
    this->setScreenOrientation(static_cast<ScreenOrientation::Orientation>(this->value("screenOrientation", 0).toInt()));
    this->setMonitorClipboard(this->value("monitorClipboard", false).toBool());
    this->setNetworkProxyType(static_cast<NetworkProxyType::ProxyType>(this->value("networkProxyType", NetworkProxyType::ProxyType(NetworkProxyType::HttpProxy)).toInt()));
    this->setNetworkProxyHostName(this->value("networkProxyHostName", QString()).toString());
    this->setNetworkProxyPort(this->value("networkProxyPort", 80).toUInt());
    this->setNetworkProxyUser(this->value("networkProxyUser", QString()).toString());
    this->setNetworkProxyPassword(QString(QByteArray::fromBase64(this->value("networkProxyPassword", QByteArray()).toByteArray())));
    this->setNetworkProxy();
    this->endGroup();
}

void Settings::saveSettings() {
    this->beginGroup("Media");
    this->setValue("youtubePlaybackQuality", this->toVariantList(m_youtubePlaybackSet));
    this->setValue("youtubeDownloadQuality", this->toVariantList(m_youtubeDownloadSet));
    this->setValue("dailymotionPlaybackQuality", this->toVariantList(m_dailymotionPlaybackSet));
    this->setValue("dailymotionDownloadQuality", this->toVariantList(m_dailymotionDownloadSet));
    this->setValue("vimeoPlaybackQuality", this->toVariantList(m_vimeoPlaybackSet));
    this->setValue("vimeoDownloadQuality", this->toVariantList(m_vimeoDownloadSet));
    this->setValue("uploadAttribute", this->uploadAttribute());
    this->setValue("mediaPlayer", this->mediaPlayer());
#ifdef MEEGO_EDITION_HARMATTAN
    this->setValue("backgroundPlaybackEnabled", this->backgroundPlaybackEnabled());
#endif
#ifndef Q_OS_SYMBIAN
    this->setValue("mediaPlayerCommand", this->mediaPlayerCommand());
#endif
    this->setValue("languageFilter", this->languageFilter());
    this->endGroup();

    this->beginGroup("Transfers");
    this->setValue("startTransfersAutomatically", this->startTransfersAutomatically());
    this->setValue("maximumConcurrentTransfers", this->maximumConcurrentTransfers());
    this->setValue("downloadPath", this->downloadPath());
    this->endGroup();

    this->beginGroup("Search");
    this->setValue("safeSearch", this->safeSearch());
    this->setValue("searchSite", Services::VideoService(this->searchSite()));
    this->setValue("showSearchSite", this->showSearchSite());
    this->setValue("searchOrder", Queries::QueryOrder(this->searchOrder()));
    this->setValue("showSearchOrder", this->showSearchOrder());
    this->setValue("searchTime", Queries::TimeFilter(this->searchTime()));
    this->setValue("showSearchTime", this->showSearchTime());
    this->setValue("searchDuration", Queries::DurationFilter(this->searchDuration()));
    this->setValue("showSearchDuration", this->showSearchDuration());
    this->setValue("searchType", Queries::QueryType(this->searchType()));
    this->setValue("showSearchType", this->showSearchType());
    this->setValue("showSearchLanguage", this->showSearchLanguage());
    this->endGroup();

#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    this->beginGroup("Events");
    this->setValue("eventFeedEnabled", (this->eventFeedEnabled()) && (this->eventFeedInstalled()));
    this->endGroup();
#endif

#if (defined (Q_OS_SYMBIAN)) || (defined (MEEGO_EDITITION_HARMATTAN))
    this->beginGroup("Appearance");
    this->setValue("theme", this->theme());
    this->setValue("activeColor", this->activeColor());
    this->setValue("activeColorString", this->activeColorString());
    this->endGroup();
#endif

    this->beginGroup("System");
    this->setValue("language", this->language());
    this->setValue("screenOrientation", ScreenOrientation::Orientation(this->screenOrientation()));
    this->setValue("monitorClipboard", this->monitorClipboard());
    this->setValue("networkProxyType", NetworkProxyType::ProxyType(this->networkProxyType()));
    this->setValue("networkProxyHostName", this->networkProxyHostName());
    this->setValue("networkProxyPort", this->networkProxyPort());
    this->setValue("networkProxyUser", this->networkProxyUser());
    this->setValue("networkProxyPassword", this->networkProxyPassword().toUtf8().toBase64());
    this->endGroup();
}

QVariantList Settings::toVariantList(const QSet<int> &set) {
    QVariantList varList;
    QList<int> list = set.toList();

    while (!list.isEmpty()) {
        varList.append(list.takeFirst());
    }

    return varList;
}

QVariantList Settings::toVariantList(const QSet<QByteArray> &set) {
    QVariantList varList;
    QList<QByteArray> list = set.toList();

    while (!list.isEmpty()) {
        varList.append(list.takeFirst());
    }

    return varList;
}

void Settings::setYouTubePlaybackQualitySet(const QVariantList &qualityList) {
    for (int i = 0; i < qualityList.size(); i++) {
        m_youtubePlaybackSet.insert(qualityList.at(i).toInt());
    }

    m_youtubePlaybackSet.insert(18);
    emit youtubePlaybackQualityChanged(m_youtubePlaybackSet);
    emit enableYouTube1080PlaybackChanged();
    emit enableYouTube720PlaybackChanged();
    emit enableYouTube480PlaybackChanged();
    emit enableYouTube360PlaybackChanged();
}

void Settings::setYouTubeDownloadQualitySet(const QVariantList &qualityList) {
    for (int i = 0; i < qualityList.size(); i++) {
        m_youtubeDownloadSet.insert(qualityList.at(i).toInt());
    }

    m_youtubeDownloadSet.insert(18);
    emit youtubeDownloadQualityChanged(m_youtubeDownloadSet);
    emit enableYouTube1080DownloadChanged();
    emit enableYouTube720DownloadChanged();
    emit enableYouTube480DownloadChanged();
    emit enableYouTube360DownloadChanged();
}

void Settings::setDailymotionPlaybackQualitySet(const QVariantList &qualityList) {
    for (int i = 0; i < qualityList.size(); i++) {
        m_dailymotionPlaybackSet.insert(qualityList.at(i).toByteArray());
    }

    m_dailymotionPlaybackSet.insert("380");
    emit dailymotionPlaybackQualityChanged(m_dailymotionPlaybackSet);
    emit enableDailymotion720PlaybackChanged();
    emit enableDailymotion480PlaybackChanged();
}

void Settings::setDailymotionDownloadQualitySet(const QVariantList &qualityList) {
    for (int i = 0; i < qualityList.size(); i++) {
        m_dailymotionDownloadSet.insert(qualityList.at(i).toByteArray());
    }

    m_dailymotionDownloadSet.insert("380");
    emit dailymotionDownloadQualityChanged(m_dailymotionDownloadSet);
    emit enableDailymotion720DownloadChanged();
    emit enableDailymotion480DownloadChanged();
}

void Settings::setVimeoPlaybackQualitySet(const QVariantList &qualityList) {
    for (int i = 0; i < qualityList.size(); i++) {
        m_vimeoPlaybackSet.insert(qualityList.at(i).toByteArray());
    }

    m_vimeoPlaybackSet.insert("mobile");
    emit vimeoPlaybackQualityChanged(m_vimeoPlaybackSet);
    emit enableVimeo720PlaybackChanged();
    emit enableVimeo360PlaybackChanged();
}

void Settings::setVimeoDownloadQualitySet(const QVariantList &qualityList) {
    for (int i = 0; i < qualityList.size(); i++) {
        m_vimeoDownloadSet.insert(qualityList.at(i).toByteArray());
    }

    m_vimeoDownloadSet.insert("mobile");
    emit vimeoDownloadQualityChanged(m_vimeoDownloadSet);
    emit enableVimeo720DownloadChanged();
    emit enableVimeo360DownloadChanged();
}

void Settings::setEnableYouTube1080Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackSet.insert(37);
    }
    else {
        m_youtubePlaybackSet.remove(37);
    }

    emit youtubePlaybackQualityChanged(m_youtubePlaybackSet);
    emit enableYouTube1080PlaybackChanged();
}

void Settings::setEnableYouTube720Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackSet.insert(22);
    }
    else {
        m_youtubePlaybackSet.remove(22);
    }

    emit youtubePlaybackQualityChanged(m_youtubePlaybackSet);
    emit enableYouTube720PlaybackChanged();
}

void Settings::setEnableYouTube480Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackSet.insert(35);
    }
    else {
        m_youtubePlaybackSet.remove(35);
    }

    emit youtubePlaybackQualityChanged(m_youtubePlaybackSet);
    emit enableYouTube480PlaybackChanged();
}

void Settings::setEnableYouTube360Playback(bool enable) {
    if (enable) {
        m_youtubePlaybackSet.insert(34);
    }
    else {
        m_youtubePlaybackSet.remove(34);
    }

    emit youtubePlaybackQualityChanged(m_youtubePlaybackSet);
    emit enableYouTube360PlaybackChanged();
}

void Settings::setEnableYouTube1080Download(bool enable) {
    if (enable) {
        m_youtubeDownloadSet.insert(37);
    }
    else {
        m_youtubeDownloadSet.remove(37);
    }

    emit youtubeDownloadQualityChanged(m_youtubeDownloadSet);
    emit enableYouTube1080DownloadChanged();
}

void Settings::setEnableYouTube720Download(bool enable) {
    if (enable) {
        m_youtubeDownloadSet.insert(22);
    }
    else {
        m_youtubeDownloadSet.remove(22);
    }

    emit youtubeDownloadQualityChanged(m_youtubeDownloadSet);
    emit enableYouTube720DownloadChanged();
}

void Settings::setEnableYouTube480Download(bool enable) {
    if (enable) {
        m_youtubeDownloadSet.insert(35);
    }
    else {
        m_youtubeDownloadSet.remove(35);
    }

    emit youtubeDownloadQualityChanged(m_youtubeDownloadSet);
    emit enableYouTube480DownloadChanged();
}

void Settings::setEnableYouTube360Download(bool enable) {
    if (enable) {
        m_youtubeDownloadSet.insert(34);
    }
    else {
        m_youtubeDownloadSet.remove(34);
    }

    emit youtubeDownloadQualityChanged(m_youtubeDownloadSet);
    emit enableYouTube360DownloadChanged();
}

void Settings::setEnableDailymotion720Playback(bool enable) {
    if (enable) {
        m_dailymotionPlaybackSet.insert("720");
    }
    else {
        m_dailymotionPlaybackSet.remove("720");
    }

    emit dailymotionPlaybackQualityChanged(m_dailymotionPlaybackSet);
    emit enableDailymotion720PlaybackChanged();
}

void Settings::setEnableDailymotion480Playback(bool enable) {
    if (enable) {
        m_dailymotionPlaybackSet.insert("480");
    }
    else {
        m_dailymotionPlaybackSet.remove("480");
    }

    emit dailymotionPlaybackQualityChanged(m_dailymotionPlaybackSet);
    emit enableDailymotion480PlaybackChanged();
}

void Settings::setEnableDailymotion720Download(bool enable) {
    if (enable) {
        m_dailymotionDownloadSet.insert("720");
    }
    else {
        m_dailymotionDownloadSet.remove("720");
    }

    emit dailymotionDownloadQualityChanged(m_dailymotionDownloadSet);
    emit enableDailymotion720DownloadChanged();
}

void Settings::setEnableDailymotion480Download(bool enable) {
    if (enable) {
        m_dailymotionDownloadSet.insert("480");
    }
    else {
        m_dailymotionDownloadSet.remove("480");
    }

    emit dailymotionDownloadQualityChanged(m_dailymotionDownloadSet);
    emit enableDailymotion480DownloadChanged();
}

void Settings::setEnableVimeo720Playback(bool enable) {
    if (enable) {
        m_vimeoPlaybackSet.insert("hd");
    }
    else {
        m_vimeoPlaybackSet.remove("hd");
    }

    emit vimeoPlaybackQualityChanged(m_vimeoPlaybackSet);
    emit enableVimeo720PlaybackChanged();
}

void Settings::setEnableVimeo360Playback(bool enable) {
    if (enable) {
        m_vimeoPlaybackSet.insert("sd");
    }
    else {
        m_vimeoPlaybackSet.remove("sd");
    }

    emit vimeoPlaybackQualityChanged(m_vimeoPlaybackSet);
    emit enableVimeo360PlaybackChanged();
}

void Settings::setEnableVimeo720Download(bool enable) {
    if (enable) {
        m_vimeoDownloadSet.insert("hd");
    }
    else {
        m_vimeoDownloadSet.remove("hd");
    }

    emit vimeoDownloadQualityChanged(m_vimeoDownloadSet);
    emit enableVimeo720DownloadChanged();
}

void Settings::setEnableVimeo360Download(bool enable) {
    if (enable) {
        m_vimeoDownloadSet.insert("sd");
    }
    else {
        m_vimeoDownloadSet.remove("sd");
    }

    emit vimeoDownloadQualityChanged(m_vimeoDownloadSet);
    emit enableVimeo360DownloadChanged();

}

void Settings::setStartTransfersAutomatically(bool automatic) {
    if (automatic != this->startTransfersAutomatically()) {
        m_autoTransfers = automatic;
        emit startTransfersAutomaticallyChanged(automatic);
    }
}

void Settings::setMaximumConcurrentTransfers(int maximum) {
    int oldMaximum = this->maximumConcurrentTransfers();

    if ((maximum != oldMaximum) && (maximum > 0) && (maximum <= 3)) {
        m_maxConcurrentTransfers = maximum;
        emit maximumConcurrentTransfersChanged(oldMaximum, maximum);
    }
}

void Settings::setDownloadPath(const QString &path) {
    m_path = path;

    if (!m_path.endsWith("/")) {
        m_path.append("/");
    }

    QDir pathCheck(m_path + ".thumbnails/");

    if (pathCheck.exists(m_path + "thumbnails/")) {
        pathCheck.rename(m_path + "thumbnails/", pathCheck.path());
    }

    pathCheck.mkpath(pathCheck.path());

    emit downloadPathChanged(m_path);
}

void Settings::setSafeSearch(bool safe) {
    if (safe != this->safeSearch()) {
        m_safeSearch = safe;
        emit safeSearchChanged(safe);
    }
}

void Settings::setSearchSite(Services::VideoService site) {
    if (site != this->searchSite()) {
        m_searchSite = site;
        emit searchSiteChanged(site);
    }
}

void Settings::setShowSearchSite(bool show) {
    if (show != this->showSearchSite()) {
        m_showSearchSite = show;
        emit showSearchSiteChanged(show);
    }
}

void Settings::setSearchOrder(Queries::QueryOrder order) {
    if (order != this->searchOrder()) {
        m_searchOrder = order;
        emit searchOrderChanged(order);
    }
}

void Settings::setShowSearchOrder(bool show) {
    if (show != this->showSearchOrder()) {
        m_showSearchOrder = show;
        emit showSearchOrderChanged(show);
    }
}

void Settings::setSearchTime(Queries::TimeFilter time) {
    if (time != this->searchTime()) {
        m_searchTime = time;
        emit searchTimeChanged(time);
    }
}

void Settings::setShowSearchTime(bool show) {
    if (show != this->showSearchTime()) {
        m_showSearchTime = show;
        emit showSearchTimeChanged(show);
    }
}

void Settings::setSearchDuration(Queries::DurationFilter duration) {
    if (duration != this->searchDuration()) {
        m_searchDuration = duration;
        emit searchDurationChanged(duration);
    }
}

void Settings::setShowSearchDuration(bool show) {
    if (show != this->showSearchDuration()) {
        m_showSearchDuration = show;
        emit showSearchDurationChanged(show);
    }
}

void Settings::setSearchType(Queries::QueryType type) {
    if (type != this->searchType()) {
        m_searchType = type;
        emit searchTypeChanged(type);
    }
}

void Settings::setShowSearchType(bool show) {
    if (show != this->showSearchType()) {
        m_showSearchType = show;
        emit showSearchTypeChanged(show);
    }
}

void Settings::setLanguageFilter(const QString &language) {
    if (language != this->languageFilter()) {
        m_languageFilter = language;
        emit languageFilterChanged(language);
    }
}

void Settings::setShowSearchLanguage(bool show) {
    if (show != this->showSearchLanguage()) {
        m_showSearchLanguage = show;
        emit showSearchLanguageChanged(show);
    }
}

void Settings::setTheme(const QString &theme) {
    if (theme != this->theme()) {
        m_theme = theme;
        emit themeChanged(theme);
    }
}

void Settings::setActiveColor(const QString &color) {
    if (color != this->activeColor()) {
        m_activeColor = color;
        emit activeColorChanged(color);
    }
}

void Settings::setActiveColorString(const QString &colorString) {
    if (colorString != this->activeColorString()) {
        m_activeColorString = colorString;
        emit activeColorStringChanged(colorString);
    }
}

void Settings::setLanguage(const QString &lang) {
    if (lang != this->language()) {
        m_language = lang;
        emit languageChanged(lang);
    }
}

void Settings::setUploadAttribute(const QString &attribute) {
    if (attribute != this->uploadAttribute()) {
        m_attribute = attribute;
        emit uploadAttributeChanged(attribute);
    }
}

void Settings::setMediaPlayer(const QString &player) {
    if (player != this->mediaPlayer()) {
        m_mediaplayer = player;
        emit mediaPlayerChanged(player);
    }
}

void Settings::setMediaPlayerCommand(const QString &command) {
    if (command != this->mediaPlayerCommand()) {
        m_command = command;
        emit mediaPlayerCommandChanged(command);
    }
}

void Settings::setBackgroundPlaybackEnabled(bool enabled) {
    if (enabled != this->backgroundPlaybackEnabled()) {
        m_backgroundPlaybackEnabled = enabled;
        emit backgroundPlaybackEnabledChanged(enabled);
    }
}

void Settings::setScreenOrientation(ScreenOrientation::Orientation orientation) {
    if (orientation != this->screenOrientation()) {
        m_orientation = orientation;
        emit screenOrientationChanged(orientation);
    }
}

void Settings::setMonitorClipboard(bool monitor) {
    if (monitor != this->monitorClipboard()) {
        m_monitorClipboard = monitor;
        emit monitorClipboardChanged(monitor);
    }
}

QString Settings::networkProxyTypeString() const {
    switch (this->networkProxyType()) {
    case NetworkProxyType::Socks5Proxy:
        return QString("SOCKS5");
    case NetworkProxyType::HttpProxy:
        return QString("HTTP");
    case NetworkProxyType::HttpCachingProxy:
        return QString("HTTP %1").arg(tr("caching"));
    default:
        return QString("HTTP");
    }
}

void Settings::setNetworkProxyType(NetworkProxyType::ProxyType proxyType) {
    if (proxyType != this->networkProxyType()) {
        m_proxyType = proxyType;
        emit networkProxyTypeChanged(proxyType);
        emit networkProxyTypeStringChanged(this->networkProxyTypeString());
    }
}

void Settings::setNetworkProxyHostName(const QString &hostName) {
    if (hostName != this->networkProxyHostName()) {
        m_proxyHost = hostName;
        emit networkProxyHostNameChanged(hostName);
    }
}

void Settings::setNetworkProxyPort(quint16 port) {
    if (port != this->networkProxyPort()) {
        m_proxyPort = port;
        emit networkProxyPortChanged(port);
    }
}

void Settings::setNetworkProxyUser(const QString &user) {
    if (user != this->networkProxyUser()) {
        m_proxyUser = user;
        emit networkProxyUserChanged(user);
    }
}

void Settings::setNetworkProxyPassword(const QString &password) {
    if (password != this->networkProxyPassword()) {
        m_proxyPass = password;
        emit networkProxyPasswordChanged(password);
    }
}

void Settings::setNetworkProxy() {
    QNetworkProxy proxy;

    if (!this->networkProxyHostName().isEmpty()) {
        proxy.setHostName(this->networkProxyHostName());
        proxy.setType(QNetworkProxy::ProxyType(this->networkProxyType()));

        if (this->networkProxyPort() > 0) {
            proxy.setPort(this->networkProxyPort());
        }
        else {
            proxy.setPort(80);
        }

        if ((!this->networkProxyUser().isEmpty()) && (!this->networkProxyPassword().isEmpty())) {
            proxy.setUser(this->networkProxyUser());
            proxy.setPassword(this->networkProxyPassword());
        }
    }

    QNetworkProxy::setApplicationProxy(proxy);
}

void Settings::onNetworkProxyAuthenticationRequested(const QNetworkProxy &proxy, QAuthenticator *authenticator) {
    Q_UNUSED(proxy)

    authenticator->setUser(this->networkProxyUser());
    authenticator->setPassword(this->networkProxyPassword());
}

void Settings::setEventFeedEnabled(bool enabled) {
    if (enabled != this->eventFeedEnabled()) {
        m_eventFeedEnabled = enabled;
        emit eventFeedEnabledChanged(enabled);
    }
}

bool Settings::eventFeedInstalled() const {
#ifdef Q_WS_MAEMO_5
    QDir dir;

    return dir.exists("/opt/hildonevents/bin/hildonevents");
#elif defined (MEEGO_EDITION_HARMATTAN)
    return true;
#else
    return false;
#endif
}

void Settings::setCuteTubePlayerEnabled() {
#ifdef Q_OS_SYMBIAN
    using namespace QtMobility;
    QSystemDeviceInfo info;
    m_cuteTubePlayerEnabled = (info.model() == "700") || (info.model() == "701") || (info.model() == "808");
#endif
}
