#include "database.h"
#include "utils.h"
#include <qplatformdefs.h>
#include <QFile>
#include <QDir>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QRegExp>
#include <QDateTime>
#include <QUrl>
#include <QDebug>
#include <QDesktopServices>

Database::Database(QObject *parent) :
    QObject(parent)
{
#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    QDir dir;
    dir.mkpath("/home/user/cuteTube/");
    m_database = QSqlDatabase::addDatabase("QSQLITE");
    m_database.setDatabaseName("/home/user/cuteTube/cuteTube.sqlite");
#elif (defined (Q_OS_SYMBIAN))
    m_database = QSqlDatabase::addDatabase("QSQLITE");
    m_database.setDatabaseName("cuteTube.sqlite");
#else
    QDir dir(QDesktopServices::storageLocation(QDesktopServices::HomeLocation).append("/cuteTube/"));
    dir.mkpath(dir.path());
    m_database = QSqlDatabase::addDatabase("QSQLITE");
    m_database.setDatabaseName(dir.path() + "/cuteTube.sqlite");
#endif
    initialize();
}

void Database::initialize() {
    if (m_database.open()) {
        m_database.exec("DROP TABLE IF EXISTS videoDownloads");
        m_database.exec("CREATE TABLE IF NOT EXISTS downloads (id TEXT UNIQUE, videoId TEXT, title TEXT, thumbnailUrl TEXT, downloadPath TEXT, tempFileName TEXT, fileName TEXT, service INTEGER, quality INTEGER, priority INTEGER, saveAsAudio INTEGER, rowNumber INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS youtubeAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, cookies TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS dailymotionAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, cookies TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS vimeoAccounts (name TEXT UNIQUE, username TEXT, token TEXT, secret TEXT, cookies TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS facebookAccount (name TEXT UNIQUE, token TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS twitterAccount (name TEXT UNIQUE, token TEXT, secret TEXT)");
        m_database.close();
    }
    else {
        qDebug() << m_database.lastError().text();
    }
}

void Database::restoreAccounts() {
    this->restoreYouTubeAccount();
    this->restoreDailymotionAccount();
    this->restoreVimeoAccount();
    this->restoreFacebookAccount();
    this->restoreTwitterAccount();
}

bool Database::setYouTubeUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube account details"));

        return false;
    }
}

bool Database::setYouTubeAccessToken(const QString &token) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET token = ? WHERE active = 1");
        query.addBindValue(token);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube account details"));

        return false;
    }
}

bool Database::setYouTubeCookies(const QByteArray &cookieString) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET cookies = ? WHERE active = 1");
        query.addBindValue(cookieString);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube cookies"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube cookies"));

        return false;
    }
}

void Database::restoreYouTubeAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString refresh;
        QByteArray cookieString;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, refresh, cookies FROM youtubeAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                refresh = query.value(2).toString();
                cookieString = query.value(3).toByteArray();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotYouTubeAccount(user, token, refresh);

            if (!cookieString.isEmpty()) {
                emit gotYouTubeCookies(cookieString);
            }
        }
    }
}

bool Database::setDailymotionUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion account details"));

        return false;
    }
}

bool Database::setDailymotionAccessToken(const QString &token, const QString &refresh) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET token = ?, refresh = ? WHERE active = 1");
        query.addBindValue(token);
        query.addBindValue(refresh);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion account details"));

        return false;
    }
}

bool Database::setDailymotionCookies(const QByteArray &cookieString) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET cookies = ? WHERE active = 1");
        query.addBindValue(cookieString);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion cookies"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion cookies"));

        return false;
    }
}

void Database::restoreDailymotionAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString refresh;
        QByteArray cookieString;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, refresh, cookies FROM dailymotionAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                refresh = query.value(2).toString();
                cookieString = query.value(3).toByteArray();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotDailymotionAccount(user, token, refresh);

            if (!cookieString.isEmpty()) {
                emit gotDailymotionCookies(cookieString);
            }
        }
    }
}

bool Database::setVimeoUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo account details"));

        return false;
    }
}

bool Database::setVimeoAccessToken(const QString &token, const QString &secret) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET token = ?, secret = ? WHERE active = 1");
        query.addBindValue(token);
        query.addBindValue(secret);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo account details"));

        return false;
    }
}

bool Database::setVimeoCookies(const QByteArray &cookieString) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET cookies = ? WHERE active = 1");
        query.addBindValue(cookieString);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo cookies"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo cookies"));

        return false;
    }
}

void Database::restoreVimeoAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString secret;
        QByteArray cookieString;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, secret, cookies FROM vimeoAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                secret = query.value(2).toString();
                cookieString = query.value(3).toByteArray();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotVimeoAccount(user, token, secret);

            if (!cookieString.isEmpty()) {
                emit gotVimeoCookies(cookieString);
            }
        }
    }
}

bool Database::storeFacebookAccount(const QString &token) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO facebookAccount VALUES (?, ?)");
        query.addBindValue("facebook");
        query.addBindValue(token);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store facebook account details"));
        }
        else {
            emit gotFacebookAccount(token);
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store facebook account details"));

        return false;
    }
}

bool Database::deleteFacebookAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM facebookAccount WHERE name = ?");
        query.addBindValue("facebook");

        bool success = query.exec();

        if (success) {
            emit gotFacebookAccount(QString());
            emit info(tr("facebook account deleted. Please visit the facebook website to revoke access"));
        }
        else {
            emit error(tr("Database error. Unable to delete facebook account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to delete facebook account details"));

        return false;
    }
}

void Database::restoreFacebookAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString token;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT token FROM facebookAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                token = query.value(0).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotFacebookAccount(token);
        }
    }
}

bool Database::storeTwitterAccount(const QString &token, const QString &secret) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO twitterAccount VALUES (?, ?, ?)");
        query.addBindValue("twitter");
        query.addBindValue(token);
        query.addBindValue(secret);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store twitter account details"));
        }
        else {
            emit gotTwitterAccount(token, secret);
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store twitter account details"));

        return false;
    }
}

bool Database::deleteTwitterAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM twitterAccount WHERE name = ?");
        query.addBindValue("twitter");

        bool success = query.exec();

        if (success) {
            emit gotFacebookAccount(QString());
            emit info(tr("twitter account deleted. Please visit the twitter website to revoke access"));
        }
        else {
            emit error(tr("Database error. Unable to delete twitter account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to delete twitter account details"));

        return false;
    }
}

void Database::restoreTwitterAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString token;
        QString secret;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT token, secret FROM twitterAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                token = query.value(0).toString();
                secret = query.value(1).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotTwitterAccount(token, secret);
        }
    }
}

bool Database::storeDownload(QSharedPointer<TransferItem> transfer) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR IGNORE INTO downloads VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        query.addBindValue(transfer.data()->id());
        query.addBindValue(transfer.data()->videoId());
        query.addBindValue(transfer.data()->title());
        query.addBindValue(transfer.data()->thumbnailUrl());
        query.addBindValue(transfer.data()->downloadPath());
        query.addBindValue(transfer.data()->tempFileName());
        query.addBindValue(transfer.data()->fileName());
        query.addBindValue(Services::VideoService(transfer.data()->service()));
        query.addBindValue(Qualities::VideoQuality(transfer.data()->quality()));
        query.addBindValue(Transfers::TransferPriority(transfer.data()->priority()));
        query.addBindValue(transfer.data()->saveAsAudio());
        query.addBindValue(transfer.data()->row());
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

bool Database::removeStoredDownload(const QString &id) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM downloads WHERE id = ?");
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

bool Database::updateStoredDownload(const QString &id, const QString &key, const QVariant &value) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare(QString("UPDATE downloads SET %1 = ? WHERE id = ?").arg(key));
        query.addBindValue(value);
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

QList< QSharedPointer<TransferItem> > Database::getStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT * FROM downloads ORDER BY rowNumber ASC");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                QSharedPointer<TransferItem> transfer(new TransferItem);
                transfer.data()->setId(query.value(0).toString());
                transfer.data()->setVideoId(query.value(1).toString());
                transfer.data()->setTitle(query.value(2).toString());
                transfer.data()->setThumbnailUrl(query.value(3).toUrl());
                transfer.data()->setDownloadPath(query.value(4).toString());
                transfer.data()->setTempFileName(query.value(5).toString());
                transfer.data()->setFileName(query.value(6).toString());
                transfer.data()->setService(static_cast<Services::VideoService>(query.value(7).toInt()));
                transfer.data()->setQuality(static_cast<Qualities::VideoQuality>(query.value(8).toInt()));
                transfer.data()->setPriority(static_cast<Transfers::TransferPriority>(query.value(9).toInt()));
                transfer.data()->setSaveAsAudio(query.value(10).toBool());
                transfer.data()->setRow(query.value(11).toUInt());
                transfers.append(transfer);
            }
        }

        m_database.close();
    }

    return transfers;
}
