#include "session.h"
#include <QMaemo5InformationBox>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QX11Info>

Session::Session(QObject *parent) :
    QObject(parent),
    m_settings(new Settings(this)),
    m_database(new Database(this)),
    m_launcher(new VideoLauncher(this)),
    m_transferManager(new TransferListModel(this)),
    m_transferWorker(new TransferWorker(this)),
    m_youtube(new YouTube(this)),
    m_dailymotion(new Dailymotion(this)),
    m_vimeo(new Vimeo(this)),
    m_xtube(0),
    m_sharing(new Sharing(this)),
    m_grabber(new UrlGrabber(UrlGrabber::PlaybackMode, this)),
    m_jar(new CookieJar(CookieJar::DownloadMode, this)),
    m_thumbnailDownloader(new ThumbnailDownloader(this)),
    m_cache(new QCache<QUrl, Thumbnail>),
    m_thumbnailCache(new ThumbnailCache(m_cache)),
    m_nam(new QNetworkAccessManager(this)),
    m_thread(new QThread(this)),
    m_queue(new PlaybackListModel(newThumbnailCache(), this)),
    m_progressDialog(0)
{
    networkAccessManager()->setCookieJar(cookieJar());
    transferWorker()->setNetworkAccessManager(networkAccessManager());
    transferWorker()->setYouTubeSession(youtube());
    youtube()->setNetworkAccessManager(networkAccessManager());
    dailymotion()->setNetworkAccessManager(networkAccessManager());
    vimeo()->setNetworkAccessManager(networkAccessManager());
    sharing()->setNetworkAccessManager(networkAccessManager());
    urlGrabber()->setNetworkAccessManager(networkAccessManager());
    transferManager()->setWorker(transferWorker());
    thumbnailDownloader()->setNetworkAccessManager(networkAccessManager());
    thumbnailCache()->moveToThread(m_thread);
    m_thread->start(QThread::LowestPriority);

    connect(database(), SIGNAL(gotDailymotionAccount(QString,QString,QString)), dailymotion(), SLOT(setAccount(QString,QString,QString)));
    connect(database(), SIGNAL(gotYouTubeAccount(QString,QString,QString)), youtube(), SLOT(setAccount(QString,QString,QString)));
    connect(database(), SIGNAL(gotFacebookAccount(QString)), sharing(), SLOT(setFacebookAccessToken(QString)));
    connect(database(), SIGNAL(gotTwitterAccount(QString,QString)), sharing(), SLOT(setTwitterAccount(QString,QString)));
    connect(sharing(), SIGNAL(signedInToFacebook(QString)), database(), SLOT(storeFacebookAccount(QString)));
    connect(sharing(), SIGNAL(signedInToTwitter(QString,QString)), database(), SLOT(storeTwitterAccount(QString,QString)));
    connect(sharing(), SIGNAL(signedOutOfFacebook()), database(), SLOT(deleteFacebookAccount()));
    connect(sharing(), SIGNAL(signedOutOfTwitter()), database(), SLOT(deleteTwitterAccount()));
    connect(youtube(), SIGNAL(usernameChanged(QString)), database(), SLOT(setYouTubeUsername(QString)));
    connect(youtube(), SIGNAL(accessTokenRefreshed(QString)), database(), SLOT(setYouTubeAccessToken(QString)));
    connect(dailymotion(), SIGNAL(usernameChanged(QString)), database(), SLOT(setDailymotionUsername(QString)));
    connect(dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), database(), SLOT(setDailymotionAccessToken(QString,QString)));
    connect(settings(), SIGNAL(youtubePlaybackQualityChanged(QSet<int>)), urlGrabber(), SLOT(setYouTubeQualitySet(QSet<int>)));
    connect(settings(), SIGNAL(dailymotionPlaybackQualityChanged(QSet<QByteArray>)), urlGrabber(), SLOT(setDailymotionQualitySet(QSet<QByteArray>)));
    connect(settings(), SIGNAL(vimeoPlaybackQualityChanged(QSet<QByteArray>)), urlGrabber(), SLOT(setVimeoQualitySet(QSet<QByteArray>)));
    connect(settings(), SIGNAL(safeSearchChanged(bool)), youtube(), SLOT(setSafeSearch(bool)));
    connect(settings(), SIGNAL(safeSearchChanged(bool)), dailymotion(), SLOT(setSafeSearch(bool)));
    connect(settings(), SIGNAL(downloadPathChanged(QString)), transferWorker(), SLOT(setDownloadPath(QString)));
    connect(settings(), SIGNAL(youtubeDownloadQualityChanged(QSet<int>)), transferWorker(), SLOT(setYouTubeDownloadQualitySet(QSet<int>)));
    connect(settings(), SIGNAL(dailymotionDownloadQualityChanged(QSet<QByteArray>)), transferWorker(), SLOT(setDailymotionDownloadQualitySet(QSet<QByteArray>)));
    connect(settings(), SIGNAL(vimeoDownloadQualityChanged(QSet<QByteArray>)), transferWorker(), SLOT(setVimeoDownloadQualitySet(QSet<QByteArray>)));
    connect(settings(), SIGNAL(defaultTransferStatusChanged(TransferItem::Status)), transferManager(), SLOT(setDefaultTransferStatus(TransferItem::Status)));
    connect(settings(), SIGNAL(mediaPlayerChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
    connect(settings(), SIGNAL(mediaPlayerChanged(QString)), videoLauncher(), SLOT(setMediaPlayer(QString)));
    connect(settings(), SIGNAL(mediaPlayerCommandChanged(QString)), videoLauncher(), SLOT(setCustomCommand(QString)));
    connect(transferManager(), SIGNAL(transferAdded(QSharedPointer<TransferItem>)), database(), SLOT(storeDownload(QSharedPointer<TransferItem>)));
    connect(transferManager(), SIGNAL(transferCompleted(QSharedPointer<TransferItem>)), database(), SLOT(removeStoredDownload(QSharedPointer<TransferItem>)));
    connect(transferManager(), SIGNAL(transferCancelled(QSharedPointer<TransferItem>)), database(), SLOT(removeStoredDownload(QSharedPointer<TransferItem>)));
    connect(database(), SIGNAL(gotStoredDownloads(QList< QSharedPointer<TransferItem> >)), transferManager(), SLOT(addTransfers(QList< QSharedPointer<TransferItem> >)));
    connect(database(), SIGNAL(storedDownloadRemoved(QSharedPointer<TransferItem>)), transferManager(), SLOT(onTransferCleared(QSharedPointer<TransferItem>)));
    connect(sharing(), SIGNAL(gotYouTubeIds(QStringList)), youtube(), SLOT(getVideosFromIds(QStringList)));
    connect(sharing(), SIGNAL(gotDailymotionIds(QStringList)), dailymotion(), SLOT(getVideosFromIds(QStringList)));
    connect(youtube(), SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), sharing(), SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));
    connect(dailymotion(), SIGNAL(gotVideosFromIds(QList< QSharedPointer<VideoItem> >)), sharing(), SLOT(addVideosToCache(QList< QSharedPointer<VideoItem> >)));
    connect(database(), SIGNAL(gotYouTubeCookies(QByteArray)), cookieJar(), SLOT(setYouTubeCookies(QByteArray)));
    connect(database(), SIGNAL(gotDailymotionCookies(QByteArray)), cookieJar(), SLOT(setDailymotionCookies(QByteArray)));
    connect(cookieJar(), SIGNAL(gotYouTubeCookies(QByteArray)), database(), SLOT(setYouTubeCookies(QByteArray)));
    connect(cookieJar(), SIGNAL(gotDailymotionCookies(QByteArray)), database(), SLOT(setDailymotionCookies(QByteArray)));
    connect(sharing(), SIGNAL(signedInToTwitter(QString,QString)), cookieJar(), SLOT(clearTwitterCookies()));
    connect(thumbnailCache(), SIGNAL(downloadThumbnail(QUrl)), thumbnailDownloader(), SLOT(downloadThumbnail(QUrl)));
    connect(thumbnailDownloader(), SIGNAL(gotThumbnail(QNetworkReply*)), thumbnailCache(), SLOT(onThumbnailDownloaded(QNetworkReply*)));

    connect(database(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(database(), SIGNAL(warning(QString)), this, SLOT(showNoteBanner(QString)));
    connect(database(), SIGNAL(info(QString)), this, SLOT(showNoteBanner(QString)));
    connect(sharing(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(sharing(), SIGNAL(warning(QString)), this, SLOT(showNoteBanner(QString)));
    connect(sharing(), SIGNAL(info(QString)), this, SLOT(showNoteBanner(QString)));
    connect(youtube(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(youtube(), SIGNAL(warning(QString)), this, SLOT(showNoteBanner(QString)));
    connect(youtube(), SIGNAL(info(QString)), this, SLOT(showNoteBanner(QString)));
    connect(dailymotion(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(dailymotion(), SIGNAL(warning(QString)), this, SLOT(showNoteBanner(QString)));
    connect(dailymotion(), SIGNAL(info(QString)), this, SLOT(showNoteBanner(QString)));
    connect(vimeo(), SIGNAL(error(QString)), this, SLOT(showNoteBanner(QString)));
    connect(transferManager(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(playbackQueue(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));
    connect(urlGrabber(), SIGNAL(error(QString)), this, SLOT(showNoteBanner(QString)));
    connect(cookieJar(), SIGNAL(alert(QString)), this, SLOT(showInfoBanner(QString)));

    connect(youtube(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    connect(dailymotion(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    connect(vimeo(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    connect(sharing(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));

    settings()->restoreSettings();
    database()->restoreAccounts();
    database()->getStoredDownloads();

    if (settings()->xtubeEnabled()) {
        m_xtube = new XTube(this);
        xtube()->setNetworkAccessManager(networkAccessManager());
        connect(xtube(), SIGNAL(error(QString)), this, SLOT(showNoteBanner(QString)));
    }
}

Session::~Session() {
    m_thread->quit();
}

void Session::onMediaPlayerChanged(const QString &player) {
    if (player == "cutetube") {
        disconnect(urlGrabber(), SIGNAL(gotVideoUrl(QString)), videoLauncher(), SLOT(launchVideo(QString)));
        disconnect(urlGrabber(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));

        if (m_progressDialog) {
            disconnect(urlGrabber(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
            disconnect(urlGrabber(), SIGNAL(error(QString)), m_progressDialog, SLOT(cancel()));
        }
    }
    else {
        connect(urlGrabber(), SIGNAL(gotVideoUrl(QString)), videoLauncher(), SLOT(launchVideo(QString)));
        connect(urlGrabber(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));

        if (m_progressDialog) {
            connect(urlGrabber(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
            connect(urlGrabber(), SIGNAL(error(QString)), m_progressDialog, SLOT(cancel()));
        }
    }
}

ThumbnailCache* Session::newThumbnailCache() {
    ThumbnailCache *cache = new ThumbnailCache(m_cache);
    cache->moveToThread(m_thread);
    ThumbnailDownloader *downloader = new ThumbnailDownloader;
    downloader->setNetworkAccessManager(networkAccessManager());

    connect(cache, SIGNAL(downloadThumbnail(QUrl)), downloader, SLOT(downloadThumbnail(QUrl)));
    connect(downloader, SIGNAL(gotThumbnail(QNetworkReply*)), cache, SLOT(onThumbnailDownloaded(QNetworkReply*)));
    connect(cache, SIGNAL(destroyed()), downloader, SLOT(deleteLater()));

    return cache;
}

void Session::showInfoBanner(const QString &message) {
    QMaemo5InformationBox::information(0, message);
}

void Session::showNoteBanner(const QString &message) {
    QMaemo5InformationBox::information(0, message, QMaemo5InformationBox::NoTimeout);
}

void Session::showProgressDialog(const QString &message, int numberOfOperations) {
    if (!m_progressDialog) {
        m_progressDialog = new QProgressDialog();
        m_progressDialog->setWindowTitle(tr("Please wait"));
        m_progressDialog->setCancelButtonText(QString());
        m_progressDialog->setMinimumDuration(0);
        m_progressDialog->setMinimum(0);

        connect(youtube(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
        connect(youtube(), SIGNAL(alert(QString)), m_progressDialog, SLOT(cancel()));
        connect(youtube(), SIGNAL(warning(QString)), m_progressDialog, SLOT(cancel()));
        connect(youtube(), SIGNAL(info(QString)), m_progressDialog, SLOT(cancel()));
        connect(dailymotion(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
        connect(dailymotion(), SIGNAL(alert(QString)), m_progressDialog, SLOT(cancel()));
        connect(dailymotion(), SIGNAL(warning(QString)), m_progressDialog, SLOT(cancel()));
        connect(dailymotion(), SIGNAL(info(QString)), m_progressDialog, SLOT(cancel()));
        connect(vimeo(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
        connect(vimeo(), SIGNAL(error(QString)), m_progressDialog, SLOT(cancel()));
        connect(sharing(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
        connect(sharing(), SIGNAL(alert(QString)), m_progressDialog, SLOT(cancel()));
        connect(sharing(), SIGNAL(warning(QString)), m_progressDialog, SLOT(cancel()));
        connect(sharing(), SIGNAL(info(QString)), m_progressDialog, SLOT(cancel()));

        if (settings()->mediaPlayer() != "cutetube") {
            connect(urlGrabber(), SIGNAL(busyProgressChanged(int)), m_progressDialog, SLOT(setValue(int)));
            connect(urlGrabber(), SIGNAL(error(QString)), m_progressDialog, SLOT(cancel()));
        }
    }

    m_progressDialog->setLabelText(message);
    m_progressDialog->setMaximum(numberOfOperations);
    m_progressDialog->setValue(0);
    m_progressDialog->show();
}

void Session::toggleBusy(bool busy) const {
    if (QWidget *window = currentWindow()) {
        window->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, busy);
    }
}

QWidget* Session::currentWindow() const {
    if (!m_windowStack.isEmpty()) {
        return m_windowStack.last();
    }

    return 0;
}

void Session::addWindowToStack(QWidget *window) {
    m_windowStack.append(window);
    connect(window, SIGNAL(destroyed(QObject*)), this, SLOT(removeWindowFromStack(QObject*)));
}

void Session::removeWindowFromStack(QObject *window) {
    m_windowStack.removeOne(qobject_cast<QWidget*>(window));
}

void Session::unwindWindowStack() {
    for (int i = m_windowStack.size() - 1; i > 0; i--) {
        m_windowStack.at(i)->close();
    }
}

void Session::preventDisturb(bool prevent) {
    if (m_windowStack.isEmpty()) {
        return;
    }

    Atom atom = XInternAtom(QX11Info::display() , "_HILDON_DO_NOT_DISTURB", False);

    if (prevent) {
        long state = 1;
        XChangeProperty(
                    QX11Info::display(),
                    m_windowStack.last()->winId(),
                    atom,
                    XA_INTEGER,
                    32,
                    PropModeReplace,
                    (unsigned char *) &state,
                    1);
    }
    else {
        XDeleteProperty(QX11Info::display(), m_windowStack.last()->winId(), atom);
    }
}
