#include "database.h"
#include "utils.h"
#include <QFile>
#include <QDir>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QRegExp>
#include <QDateTime>
#include <QUrl>
#include <QDebug>

Database::Database(QObject *parent) :
    QObject(parent)
{
    QDir dir;
    dir.mkpath("/home/user/cuteTube/");
    m_database = QSqlDatabase::addDatabase("QSQLITE");
    m_database.setDatabaseName("/home/user/cuteTube/cuteTube.sqlite");
    initialize();
}

void Database::initialize() {
    if (m_database.open()) {
        m_database.exec("CREATE TABLE IF NOT EXISTS videoDownloads (id TEXT UNIQUE, title TEXT, thumbnailUrl TEXT, service INTEGER, quality INTEGER, transferType INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS youtubeAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, cookies TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS dailymotionAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, cookies TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS facebookAccount (name TEXT UNIQUE, token TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS twitterAccount (name TEXT UNIQUE, token TEXT, secret TEXT)");
        m_database.close();
    }
    else {
        qDebug() << m_database.lastError().text();
    }
}

void Database::restoreAccounts() {
    restoreYouTubeAccount();
    restoreDailymotionAccount();
    restoreFacebookAccount();
    restoreTwitterAccount();
}

bool Database::setYouTubeUsername(const QString &user) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE youtubeAccounts SET username = ? WHERE active = 1");
    query.addBindValue(user);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store YouTube account details"));
    }

    m_database.close();

    return success;
}

bool Database::setYouTubeAccessToken(const QString &token) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE youtubeAccounts SET token = ? WHERE active = 1");
    query.addBindValue(token);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store YouTube account details"));
    }

    m_database.close();

    return success;
}

bool Database::setYouTubeCookies(const QByteArray &cookieString) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE youtubeAccounts SET cookies = ? WHERE active = 1");
    query.addBindValue(cookieString);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store YouTube cookies"));
    }

    m_database.close();

    return success;
}

void Database::restoreYouTubeAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QString user;
    QString token;
    QString refresh;
    QByteArray cookieString;
    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT username, token, refresh, cookies FROM youtubeAccounts WHERE active = 1");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            user = query.value(0).toString();
            token = query.value(1).toString();
            refresh = query.value(2).toString();
            cookieString = query.value(3).toByteArray();
        }
    }

    m_database.close();

    if (!token.isEmpty()) {
        emit gotYouTubeAccount(user, token, refresh);

        if (!cookieString.isEmpty()) {
            emit gotYouTubeCookies(cookieString);
        }
    }
}

bool Database::setDailymotionUsername(const QString &user) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE dailymotionAccounts SET username = ? WHERE active = 1");
    query.addBindValue(user);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store Dailymotion account details"));
    }

    m_database.close();

    return success;
}

bool Database::setDailymotionAccessToken(const QString &token, const QString &refresh) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE dailymotionAccounts SET token = ?, refresh = ? WHERE active = 1");
    query.addBindValue(token);
    query.addBindValue(refresh);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store Dailymotion account details"));
    }

    m_database.close();

    return success;
}

bool Database::setDailymotionCookies(const QByteArray &cookieString) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("UPDATE dailymotionAccounts SET cookies = ? WHERE active = 1");
    query.addBindValue(cookieString);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store Dailymotion cookies"));
    }

    m_database.close();

    return success;
}

void Database::restoreDailymotionAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QString user;
    QString token;
    QString refresh;
    QByteArray cookieString;
    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT username, token, refresh, cookies FROM dailymotionAccounts WHERE active = 1");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            user = query.value(0).toString();
            token = query.value(1).toString();
            refresh = query.value(2).toString();
            cookieString = query.value(3).toByteArray();
        }
    }

    m_database.close();

    if (!token.isEmpty()) {
        emit gotDailymotionAccount(user, token, refresh);

        if (!cookieString.isEmpty()) {
            emit gotDailymotionCookies(cookieString);
        }
    }
}

bool Database::storeFacebookAccount(const QString &token) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("INSERT OR REPLACE INTO facebookAccount VALUES (?, ?)");
    query.addBindValue("facebook");
    query.addBindValue(token);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store facebook account details"));
    }
    else {
        emit gotFacebookAccount(token);
    }

    m_database.close();

    return success;
}

bool Database::deleteFacebookAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("DELETE FROM facebookAccount WHERE name = ?");
    query.addBindValue("facebook");

    bool success = query.exec();

    if (success) {
        emit gotFacebookAccount(QString());
        emit info(tr("facebook account deleted. Please visit the facebook website to revoke access"));
    }
    else {
        emit warning(tr("Database error. Unable to delete facebook account details"));
    }

    m_database.close();

    return success;
}

void Database::restoreFacebookAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QString token;
    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT token FROM facebookAccount");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            token = query.value(0).toString();
        }
    }

    m_database.close();

    if (!token.isEmpty()) {
        emit gotFacebookAccount(token);
    }
}

bool Database::storeTwitterAccount(const QString &token, const QString &secret) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("INSERT OR REPLACE INTO twitterAccount VALUES (?, ?, ?)");
    query.addBindValue("twitter");
    query.addBindValue(token);
    query.addBindValue(secret);

    bool success = query.exec();

    if (!success) {
        emit warning(tr("Database error. Unable to store twitter account details"));
    }
    else {
        emit gotTwitterAccount(token, secret);
    }

    m_database.close();

    return success;
}

bool Database::deleteTwitterAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("DELETE FROM twitterAccount WHERE name = ?");
    query.addBindValue("twitter");

    bool success = query.exec();

    if (success) {
        emit gotFacebookAccount(QString());
        emit info(tr("twitter account deleted. Please visit the twitter website to revoke access"));
    }
    else {
        emit warning(tr("Database error. Unable to delete twitter account details"));
    }

    m_database.close();

    return success;
}

void Database::restoreTwitterAccount() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QString token;
    QString secret;
    QSqlQuery query;
    QSqlRecord record;
    query.exec("SELECT token, secret FROM twitterAccount");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            token = query.value(0).toString();
            secret = query.value(1).toString();
        }
    }

    m_database.close();

    if (!token.isEmpty()) {
        emit gotTwitterAccount(token, secret);
    }
}

void Database::setDownloadPath(const QString &path) {
    m_path = path;
}

void Database::setDefaultTransferStatus(int status) {
    m_transferStatus = static_cast<TransferItem::Status>(status);
}

bool Database::storeDownload(QSharedPointer<TransferItem> transfer) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    bool success = false;
    QSqlQuery query;
    query.prepare("INSERT OR IGNORE INTO videoDownloads VALUES (?, ?, ?, ?, ?, ?)");
    query.addBindValue(transfer.data()->id());
    query.addBindValue(transfer.data()->title());
    query.addBindValue(transfer.data()->thumbnailUrl());
    query.addBindValue(Services::VideoService(transfer.data()->service()));
    query.addBindValue(Qualities::VideoQuality(transfer.data()->quality()));
    query.addBindValue(TransferItem::TransferType(transfer.data()->transferType()));
    success = query.exec();
    m_database.close();

    return success;
}

void Database::removeStoredDownload(QSharedPointer<TransferItem> transfer) {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    query.prepare("DELETE FROM videoDownloads WHERE id = ?");
    query.addBindValue(transfer.data()->id());
    bool success = query.exec();
    m_database.close();

    if (success) {
        emit storedDownloadRemoved(transfer);
    }
}

void Database::getStoredDownloads() {
    if (!m_database.isOpen()) {
        m_database.open();
    }

    QSqlQuery query;
    QSqlRecord record;
    QList< QSharedPointer<TransferItem> > transfers;
    query.exec("SELECT * FROM videoDownloads");
    record = query.record();

    if (record.count() > 0) {
        while (query.next()) {
            QSharedPointer<TransferItem> transfer(new TransferItem);
            transfer.data()->setId(query.value(0).toString());
            transfer.data()->setTitle(query.value(1).toString());
            transfer.data()->setThumbnailUrl(query.value(2).toUrl());
            transfer.data()->setService(static_cast<Services::VideoService>(query.value(3).toInt()));
            transfer.data()->setQuality(static_cast<Qualities::VideoQuality>(query.value(4).toInt()));
            transfer.data()->setTransferType(static_cast<TransferItem::TransferType>(query.value(5).toInt()));
            transfers.append(transfer);
        }
    }

    m_database.close();

    if (!transfers.isEmpty()) {
        emit gotStoredDownloads(transfers);
    }
}
