#ifndef DAILYMOTION_H
#define DAILYMOTION_H

#include "enums.h"
#include "feedurls.h"
#include "videoitem.h"
#include "playlistitem.h"
#include "groupitem.h"
#include "useritem.h"
#include "commentitem.h"
#include "newplaylist.h"
#include "newcomment.h"
#include "videometadata.h"
#include <QObject>
#include <QUrl>
#include <QString>
#include <QVariantMap>

class QNetworkAccessManager;
class QNetworkReply;

class Dailymotion : public QObject {
    Q_OBJECT

public:
    explicit Dailymotion(QObject *parent = 0);
    ~Dailymotion();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString username() const { return m_user; }
    inline bool userSignedIn() const { return !accessToken().isEmpty(); }
    inline QString uploadsFeed() const { return DAILYMOTION_UPLOADS_FEED; }
    inline QString favouritesFeed() const { return DAILYMOTION_FAVOURITES_FEED; }
    inline QString playlistsFeed() const { return DAILYMOTION_PLAYLISTS_FEED; }
    inline QString groupsFeed() const { return DAILYMOTION_GROUPS_FEED; }
    inline QString subscriptionsFeed() const { return DAILYMOTION_SUBSCRIPTIONS_FEED; }
    inline QString newSubscriptionVideosFeed() const { return DAILYMOTION_NEW_SUBSCRIPTION_VIDEOS_FEED; }
    QNetworkReply* createReply(QString feed, int offset = 0);
    QNetworkReply* createSearchReply(Queries::QueryType queryType,
                                     const QString &query,
                                     int offset = 1,
                                     Queries::QueryOrder order = Queries::Relevance,
                                     const QString &language = QString());
    QUrl authUrl() const;
    Q_INVOKABLE void signIn(const QString &displayName, const QUrl &response);
    Q_INVOKABLE void signIn(const QString &displayName, const QString &user, const QString &pass);
    inline bool safeSearch() const { return m_safeSearch; }
    inline QList< QSharedPointer<PlaylistItem> > * playlists() const { return m_playlistCache; }
    inline QList< QSharedPointer<UserItem> > * subscriptions() const { return m_subscriptionCache; }
    inline QList< QSharedPointer<GroupItem> > * groups() const { return m_groupCache; }
    inline bool playlistsLoaded() const { return m_playlistCacheLoaded; }
    inline bool subscriptionsLoaded() const { return m_subscriptionCacheLoaded; }
    inline bool groupsLoaded() const { return m_groupCacheLoaded; }
    inline bool busy() const { return m_busy; }

public slots:
    void getPlaylists(int offset = 1);
    void getSubscriptions(int offset = 1);
    void getGroups(int offset = 1);
    inline void setSafeSearch(bool safe) { m_safeSearch = safe; }
    void signOut();
    void setAccount(const QString &user = QString(), const QString &token = QString(), const QString &refresh = QString());
    void deleteFromUploads(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deleteFromUploads(QSharedPointer<VideoItem> video, bool appendToList = true);
    void addToFavourites(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void addToFavourites(QSharedPointer<VideoItem> video, bool appendToList = true);
    void deleteFromFavourites(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deleteFromFavourites(QSharedPointer<VideoItem> video, bool appendToList = true);
    void addToPlaylist(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >(), const QString &playlistId = QString());
    void addToPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId = QString(), bool appendToList = true);
    void deleteFromPlaylist(QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >(), const QString &playlistId = QString());
    void deleteFromPlaylist(QSharedPointer<VideoItem> video, const QString &playlistId = QString(), bool appendToList = true);
    void createPlaylist(const NewPlaylist &playlist = NewPlaylist(), QSharedPointer<VideoItem> = QSharedPointer<VideoItem>(), QList< QSharedPointer<VideoItem> > videos = QList< QSharedPointer<VideoItem> >());
    void deletePlaylist(const QString &id = QString());
    void joinGroup(QSharedPointer<GroupItem> group = QSharedPointer<GroupItem>());
    void leaveGroup(QSharedPointer<GroupItem> group = QSharedPointer<GroupItem>());
    void addComment(const NewComment &comment = NewComment());
    void subscribe(QSharedPointer<UserItem> user = QSharedPointer<UserItem>());
    void unsubscribe(QSharedPointer<UserItem> user = QSharedPointer<UserItem>());
    void updateVideoMetadata(QSharedPointer<VideoItem> video = QSharedPointer<VideoItem>(), const VideoMetadata &metadata = VideoMetadata());
    void refreshAccessToken();
    void getFullVideo(QString id);
    void getCurrentUserProfile();
    void getUserProfile(const QString &id);
    void subscribedToChannel(QSharedPointer<UserItem> user = QSharedPointer<UserItem>());
    void getVideosFromIds(QStringList ids);
    void getVideoMetadata(const QString &id = QString());
    void memberOfGroup(QSharedPointer<GroupItem> group = QSharedPointer<GroupItem>());

private slots:
    void checkIfSignedIn();
    void checkIfSignedOut();
    void postRequest(const QUrl &url, const QByteArray &data = QByteArray());
    void deleteRequest(const QUrl &url);
    void postFinished();
    void onVideoDeleted();
    void onVideoMetadataUpdated();
    void onAddedToFavourites(const QVariantMap &response);
    void onDeletedFromFavourites();
    void onAddedToPlaylist(const QVariantMap &response);
    void onDeletedFromPlaylist();
    void onPlaylistCreated(const QVariantMap &response);
    void onPlaylistDeleted();
    void onSubscribed();
    void onUnsubscribed();
    void onGroupJoined();
    void onGroupLeft();
    void onCommentAdded(const QVariantMap &response);
    void onVideoActionError(const QString &error);
    void onPlaylistActionError(const QString &error);
    void onGroupActionError(const QString &error);
    void onUserActionError(const QString &error);
    void onCommentActionError(const QString &error);
    void checkFullVideo();
    void checkTokenRefresh();
    void checkCurrentUserProfile();
    void checkUserProfile();
    void addPlaylists();
    void addSubscriptions();
    void addGroups();
    void checkFullVideos();
    void checkCachePlaylist();
    void checkVideoMetadata();
    void checkAddedComment();

private:
    inline void setUsername(const QString &user) { m_user = user; emit usernameChanged(user); }
    inline QString accessToken() const { return m_token; }
    inline void setAccessToken(const QString &token) { m_token = token; emit userSignedInChanged(); }
    inline QString refreshToken() const { return m_refreshToken; }
    inline void setRefreshToken(const QString &token) { m_refreshToken = token; }
    void clearCache();
    void setPlaylistsLoaded(bool loaded) { m_playlistCacheLoaded = loaded; emit playlistsLoadedChanged(); }
    void setSubscriptionsLoaded(bool loaded) { m_subscriptionCacheLoaded = loaded; emit subscriptionsLoadedChanged(); }
    void setGroupsLoaded(bool loaded) { m_groupCacheLoaded = loaded; emit groupsLoadedChanged(); }
    void addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist);
    bool removePlaylistFromCache(const QString &id);
    void addNewSubscriptionToCache(QSharedPointer<UserItem> user);
    bool removeSubscriptionFromCache(const QString &id);
    void getPlaylistForCache(const QString &id);
    void updatePlaylistVideoCount(const QString &id, int change);
    void addNewGroupToCache(QSharedPointer<GroupItem> group);
    bool removeGroupFromCache(const QString &id);
    void getAddedComment(const QString &id);
    inline void setBusy(bool busy) { m_busy = busy; emit busyChanged(); }

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged();
    void info(const QString &message);
    void warning(const QString &message);
    void usernameChanged(const QString &username);
    void userSignedInChanged();
    void postSuccessful(const QVariantMap &response);
    void postFailed(const QString &error);
    void deletedFromUploads(QSharedPointer<VideoItem> video);
    void addedToFavourites(QSharedPointer<VideoItem> video);
    void deletedFromFavourites(QSharedPointer<VideoItem> video);
    void addedToPlaylist(const QString &id, QSharedPointer<VideoItem> video);
    void deletedFromPlaylist(const QString &id, QSharedPointer<VideoItem> video);
    void commentAdded(QSharedPointer<CommentItem> comment);
    void groupJoined(QSharedPointer<GroupItem> group);
    void groupLeft(QSharedPointer<GroupItem> group);
    void signedOut(const QString &user);
    void signedIn(const QString &displayName, const QString &token, const QString &refresh);
    void newAccountSet();
    void accessTokenRefreshed(const QString &token, const QString &refresh);
    void refreshError();
    void gotVideo(QSharedPointer<VideoItem> video);
    void gotUser(QSharedPointer<UserItem> user);
    void playlistsLoadedChanged();
    void subscriptionsLoadedChanged();
    void groupsLoadedChanged();
    void allPlaylistsLoaded();
    void allSubscriptionsLoaded();
    void allGroupsLoaded();
    void playlistAddedToCache(int row);
    void playlistRemovedFromCache(int row);
    void playlistUpdated(int row);
    void subscriptionAddedToCache(int row);
    void subscriptionRemovedFromCache(int row);
    void groupAddedToCache(int row);
    void groupRemovedFromCache(int row);
    void gotVideosFromIds(QList< QSharedPointer<VideoItem> > videos);
    void gotVideoMetadata(const VideoMetadata &metadata);

private:
    QNetworkAccessManager *m_nam;
    QString m_clientId;
    QString m_clientSecret;
    QString m_user;
    QString m_token;
    QString m_refreshToken;
    bool m_safeSearch;
    QList< QSharedPointer<VideoItem> > m_videoActionList;
    QSharedPointer<VideoItem> m_videoAction;
    NewPlaylist m_playlistAction;
    QSharedPointer<GroupItem> m_groupAction;
    QSharedPointer<UserItem> m_userAction;
    NewComment m_commentAction;
    QString m_playlistActionId;
    VideoMetadata m_metadataAction;
    int m_actionsProcessed;
    QList< QSharedPointer<PlaylistItem> > *m_playlistCache;
    QList< QSharedPointer<UserItem> > *m_subscriptionCache;
    QList< QSharedPointer<GroupItem> > *m_groupCache;
    bool m_playlistCacheLoaded;
    bool m_subscriptionCacheLoaded;
    bool m_groupCacheLoaded;
    bool m_busy;
    QHash<int, QString> m_queryOrders;
};

#endif // DAILYMOTION_H
