#include "commentdelegate.h"
#include "abstractcommentlistmodel.h"
#include <QApplication>
#include <QMouseEvent>

CommentDelegate::CommentDelegate(QListView *parent) :
    QStyledItemDelegate(parent),
    m_view(parent),
    m_avatarPressed(false),
    m_arrowPressed(false)
{
}

void CommentDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const {
    int indent = index.data(AbstractCommentListModel::IndentRole).toInt();

    if (((option.state) & (QStyle::State_Selected)) && (!m_avatarPressed) && (!m_arrowPressed)) {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundPressed.png"));
    }

    QRect backgroundRect = option.rect;
    backgroundRect.setLeft(backgroundRect.left() + 5 + (indent * 20));
    backgroundRect.setRight(backgroundRect.right() - 5);
    backgroundRect.setTop(backgroundRect.top() + 5);

    QRect artistRect = backgroundRect;
    artistRect.setLeft(artistRect.left() + 60);
    artistRect.setRight(artistRect.right() - 58);

    QFont oldFont = painter->font();
    QFont newFont;
    newFont.setPixelSize(18);
    painter->setFont(newFont);

    QString artistText = QString("%1 - %2").arg(index.data(AbstractCommentListModel::ArtistRole).toString(), index.data(AbstractCommentListModel::DateRole).toString());

    painter->drawText(artistRect, Qt::AlignTop | Qt::TextWordWrap, artistText);

    QPen oldPen = painter->pen();
    QPen newPen = oldPen;
    newPen.setColor(QColor("#4d4d4d"));
    painter->setPen(newPen);
    painter->setFont(oldFont);

    if ((option.state) & (QStyle::State_Selected) && (m_avatarPressed)) {
        painter->setOpacity(0.5);
    }

    QImage avatar = index.data(AbstractCommentListModel::ArtistAvatarRole).value<QImage>();

    if (!avatar.isNull()) {
        painter->drawImage(backgroundRect.left(), backgroundRect.top(), avatar);
    }

    painter->drawRect(backgroundRect.left(), backgroundRect.top(), 50, 50);

    if ((!index.data(AbstractCommentListModel::ReplyIdRole).toString().isEmpty()) && (indent == 0)) {
        if ((option.state) & (QStyle::State_Selected) && (m_arrowPressed)) {
            painter->setOpacity(0.5);
        }

        painter->drawImage(backgroundRect.right() - 43, backgroundRect.top() - 10, QImage("/usr/share/icons/hicolor/48x48/hildon/rss_reader_move_up.png"));
    }

    painter->setOpacity(1);

    QRect bodyRect = backgroundRect;
    bodyRect.setTop(bodyRect.top() + 55);
    bodyRect.setLeft(bodyRect.left() + 55);

    painter->drawText(bodyRect, Qt::TextWordWrap, index.data(AbstractCommentListModel::BodyRole).toString());

    painter->setPen(oldPen);
}

bool CommentDelegate::editorEvent(QEvent *event, QAbstractItemModel *model, const QStyleOptionViewItem &option, const QModelIndex &index) {
    Q_UNUSED(model);

    if (event->type() == QEvent::MouseButtonPress) {
        QMouseEvent *mouse = static_cast<QMouseEvent*>(event);

        QRect avatarRect = option.rect;
        avatarRect.setLeft(avatarRect.left() + 5 + (index.data(AbstractCommentListModel::IndentRole).toInt() * 20));
        avatarRect.setTop(avatarRect.top() + 5);
        avatarRect.setSize(QSize(50, 50));

        if (avatarRect.contains(mouse->pos())) {
            m_avatarPressed = true;
        }
        else if (!index.data(AbstractCommentListModel::LoadingRole).toBool()) {
            QRect arrowRect = option.rect;
            arrowRect.setLeft(arrowRect.right() - 53);
            arrowRect.setSize(QSize(48, 48));

            if (arrowRect.contains(mouse->pos())) {
                m_arrowPressed = true;
            }
        }

        return true;
    }

    if (event->type() == QEvent::MouseButtonRelease) {
        QMouseEvent *mouse = static_cast<QMouseEvent*>(event);

        if (m_avatarPressed) {
            QRect avatarRect = option.rect;
            avatarRect.setLeft(avatarRect.left() + 5 + (index.data(AbstractCommentListModel::IndentRole).toInt() * 10));
            avatarRect.setTop(avatarRect.top() + 5);
            avatarRect.setSize(QSize(50, 50));

            if (avatarRect.contains(mouse->pos())) {
                emit showUserChannel(index);
            }
        }
        else if ((m_arrowPressed) && (!index.data(AbstractCommentListModel::LoadingRole).toBool())) {
            QRect arrowRect = option.rect;
            arrowRect.setLeft(arrowRect.right() - 48);
            arrowRect.setSize(QSize(48, 48));

            if (arrowRect.contains(mouse->pos())) {
                emit getInitialComment(index.row());
            }
        }

        m_avatarPressed = false;
        m_arrowPressed = false;

        return true;
    }

    return false;
}

QSize CommentDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const {
    return QSize(m_view->width(), option.fontMetrics.boundingRect(option.rect, Qt::TextWordWrap, index.data(AbstractCommentListModel::BodyRole).toString()).height() + 100);
}
