#include "youtubevideolistwindow.h"
#include "youtubevideoinfowindow.h"
#include "youtubeplaylistdialog.h"
#include "videoselectionwindow.h"
#include "videolistdelegate.h"
#include <QMenuBar>

YouTubeVideoListWindow::YouTubeVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_deleteFromUploadsAction(menuBar()->addAction(tr("Delete from uploads"), this, SLOT(deleteVideosFromUploads()))),
    m_addToFavouritesAction(menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_deleteFromFavouritesAction(menuBar()->addAction(tr("Delete from favourites"), this, SLOT(deleteVideosFromFavourites()))),
    m_addToPlaylistAction(menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_watchLaterAction(menuBar()->addAction(tr("Watch later"), this, SLOT(addVideosToWatchLaterPlaylist()))),
    m_addToQueueAction(menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_deleteFromFavouritesContextAction(m_contextMenu->addAction(tr("Delete from favourites"), this, SLOT(deleteVideoFromFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_watchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterPlaylist()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));

    onOrientationChanged();
}

void YouTubeVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeVideoListWindow::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    setMenuActions();
}

void YouTubeVideoListWindow::getRelatedVideos(const QString &id) {
    m_model->getRelatedVideos(id);
    setMenuActions();
}

void YouTubeVideoListWindow::search(const QString &query, Queries::QueryOrder order, Queries::TimeFilter time, Queries::DurationFilter duration, const QString &language) {
    setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order, time, duration, language);
    setMenuActions();
}

void YouTubeVideoListWindow::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->uploads()));
    m_addToFavouritesAction->setEnabled((signedIn) && (!m_model->favourites()));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->favourites()));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled((signedIn) && (!m_model->watchLater()));
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->uploads()));
    m_addToFavouritesContextAction->setVisible((signedIn) && (!m_model->favourites()));
    m_deleteFromFavouritesContextAction->setVisible((signedIn) && (m_model->favourites()));
    m_addToPlaylistContextAction->setVisible(signedIn);
    m_watchLaterContextAction->setVisible((signedIn) && (!m_model->watchLater()));
}

void YouTubeVideoListWindow::onLoadingChanged() {
    toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void YouTubeVideoListWindow::deleteVideosFromUploads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), this);
    window->show();
}

void YouTubeVideoListWindow::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::deleteVideosFromFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideosToWatchLaterPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsAction);
}

void YouTubeVideoListWindow::deleteVideoFromUploads() {
    m_session->youtube()->deleteFromUploads(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::addVideoToFavourites() {
    m_session->youtube()->addToFavourites(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::deleteVideoFromFavourites() {
    m_session->youtube()->deleteFromFavourites(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void YouTubeVideoListWindow::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(m_model->get(m_listView->currentIndex().row()), id);
}

void YouTubeVideoListWindow::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->youtube()->createPlaylist(playlist, m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void YouTubeVideoListWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, this);
    window->show();
}

void YouTubeVideoListWindow::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
}
