#include "sharing.h"
#include "json.h"
#include "oauth.h"
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QApplication>
#include <QClipboard>
#include <QDateTime>
#include <QDesktopServices>

using namespace QtJson;
using namespace QtOAuth;

Sharing::Sharing(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_facebookClientId("385476614847489"),
    m_facebookClientSecret("711d38a113d298af9348386957c524e5"),
    m_twitterClientId("YmfKe14OCdVxn9FCL1S77A"),
    m_twitterClientSecret("evSW7txyhlvvPXkezKvJgaZ2lC97iXZc4ZLXLZxfsfc"),
    m_redirectUri("https://sites.google.com/site/marxodian/home/cutetube"),
    m_facebookTimestamp(0),
    m_videoCache(new QList< QSharedPointer<VideoItem> >),
    m_busy(false)
{
}

Sharing::~Sharing() {
    clearCache();
}

void Sharing::clearCache() {
    m_videoCache->clear();
}

QUrl Sharing::facebookAuthUrl() const {
    QUrl url("https://www.facebook.com/dialog/oauth");
    url.addQueryItem("client_id", m_facebookClientId);
    url.addQueryItem("redirect_uri", "http://www.facebook.com/connect/login_success.html");
    url.addQueryItem("response_type", "token");
    url.addQueryItem("display", "popup");
    url.addQueryItem("scope", "publish_stream,read_stream");

    return url;
}

void Sharing::signInToFacebook(const QUrl &response) {
    QUrl url(response.toString().replace('#', '?'));
    QString token = url.queryItemValue("access_token");

    if (token.isEmpty()) {
        QString error = url.queryItemValue("error");

        if (error == "access_denied") {
            emit info(tr("You have denied access to your facebook account"));
        }
        else {
            emit warning(tr("Error obtaining facebook authorisation"));
        }
    }
    else {
        emit signedInToFacebook(token);
        emit alert(tr("You are signed in to your facebook account"));
    }
}

QByteArray Sharing::getTwitterOAuthHeader(const QString &method, const QString &url, QMap<QString, QString> params) {
    params.insert("oauth_consumer_key", m_twitterClientId);
    params.insert("oauth_consumer_secret", m_twitterClientSecret);

    if (userSignedInToTwitter()) {
        params.insert("oauth_token", twitterToken());
        params.insert("oauth_token_secret", twitterTokenSecret());
    }

    return OAuth::createOAuthHeader(method, url, params);
}

void Sharing::getTwitterAuthUrl() {
    QUrl url("https://api.twitter.com/oauth/request_token");
    QMap<QString, QString> params;
    params.insert("oauth_callback", m_redirectUri);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setRawHeader("Authorization", getTwitterOAuthHeader("GET", url.toString(), params));
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkTwitterAuthUrl()));
}

void Sharing::checkTwitterAuthUrl() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QUrl url("http://twitter.com?" + reply->readAll());
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");

    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit warning(tr("Error obtaining twitter authorisation URL"));
    }
    else {
        setTwitterRequestToken(token);
        setTwitterRequestSecret(secret);
        QUrl authUrl("http://api.twitter.com/oauth/authorize");
        authUrl.addQueryItem("oauth_token", token);
        emit gotTwitterAuthUrl(authUrl);
    }

    reply->deleteLater();
}

void Sharing::signInToTwitter(const QUrl &response) {
    QString verifier = response.queryItemValue("oauth_verifier");

    if (verifier.isEmpty()) {
        emit warning(tr("Error obtaining twitter authorisation verifier"));
    }
    else {
        emit busy(tr("Signing in"));
        QUrl url("https://api.twitter.com/oauth/access_token");
        QMap<QString, QString> params;
        params.insert("oauth_token", twitterRequestToken());
        params.insert("oauth_token_secret", twitterRequestSecret());
        params.insert("oauth_verifier", verifier);
        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
        request.setRawHeader("Authorization", getTwitterOAuthHeader("GET", url.toString(), params));
        QNetworkReply *reply = networkAccessManager()->get(request);
        connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedInToTwitter()));
    }
}

void Sharing::checkIfSignedInToTwitter() {
    emit busyProgressChanged(1);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QUrl url("http://twitter.com?" + reply->readAll());
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");

    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit warning(tr("Error obtaining twitter authorisation"));
    }
    else {
        setTwitterRequestToken("");
        setTwitterRequestSecret("");
        emit signedInToTwitter(token, secret);
        emit alert(tr("You are signed in to your twitter account"));
    }

    reply->deleteLater();
}

void Sharing::setFacebookAccessToken(const QString &token) {
    m_facebookToken = token;
    emit userSignedInToFacebookChanged();
}

void Sharing::signOutOfFacebook() {
    setFacebookAccessToken(QString());
    emit signedOutOfFacebook();
    emit info(tr("Access token deleted. Please visit the facebook website to revoke access"));
}

void Sharing::setTwitterAccount(const QString &token, const QString &secret) {
    setTwitterAccessToken(token);
    setTwitterTokenSecret(secret);
    emit userSignedInToTwitterChanged();
}

void Sharing::signOutOfTwitter() {
    setTwitterAccount(QString(), QString());
    emit signedOutOfTwitter();
    emit info(tr("Access token deleted. Please visit the twitter website to revoke access"));
}

void Sharing::postToFacebook(const VideoShare &post) {
    if (!post.isEmpty()) {
        m_post = post;
    }

    QByteArray postData = "access_token=" + facebookToken().toUtf8()
            + "&message=" + m_post.message().toUtf8()
            + "&link=" + m_post.videoUrl().toUtf8()
            + "&source=" + m_post.embedUrl().toUtf8()
            + "&picture=" + m_post.thumbnailUrl().toUtf8()
            + "&name=" + m_post.title().toUtf8()
            + "&description=" + m_post.description().toUtf8();

    QNetworkRequest request(QUrl("https://graph.facebook.com/me/feed"));
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = networkAccessManager()->post(request, postData);
    connect(reply, SIGNAL(finished()), this, SLOT(facebookPostFinished()));
}

void Sharing::facebookPostFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 400) && (userSignedInToFacebook())) {
        setFacebookAccessToken(QString());
        emit signedOutOfFacebook();
        emit reauthenticateForFacebook();
        connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            emit alert(tr("Video shared to facebook"));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit warning(statusText);
        }

        disconnect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(postToFacebook()));
    }

    reply->deleteLater();
}

void Sharing::postToTwitter(const VideoShare &post) {
    if (!post.isEmpty()) {
        m_post = post;
    }

    QByteArray body = m_post.message().toUtf8().toPercentEncoding();
    QUrl url("http://api.twitter.com/1/statuses/update.json");
    QMap<QString, QString> params;
    params.insert("status", QString(body));
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("Authorization", getTwitterOAuthHeader("POST", url.toString(), params));
    QNetworkReply *reply = networkAccessManager()->post(request, "status=" + body);
    connect(reply, SIGNAL(finished()), this, SLOT(twitterPostFinished()));
}

void Sharing::twitterPostFinished() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (userSignedInToTwitter())) {
        setTwitterAccount(QString(), QString());
        emit signedOutOfTwitter();
        emit reauthenticateForTwitter();
        connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            emit alert(tr("Video shared to twitter"));
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit warning(statusText);
        }

        disconnect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(postToTwitter()));
    }

    reply->deleteLater();
}

void Sharing::shareViaEmail(QSharedPointer<VideoItem> video) {
    QDesktopServices::openUrl(QUrl(QString("mailto:?subject=%1&body=%2").arg(video.data()->title()).arg(video.data()->url().toString())));
}

void Sharing::copyToClipboard(const QUrl &url) {
    QApplication::clipboard()->setText(url.toString());
    emit alert(tr("Copied to clipboard"));
}

void Sharing::getVideos(bool forceRefresh) {
    if ((forceRefresh) || (videoRefreshIsDue())) {
        if (userSignedInToFacebook()) {
            getVideosFromFacebook();
        }
        else if (userSignedInToTwitter()) {
            getVideosFromTwitter();
        }
    }
    else if (userSignedInToFacebook() && (facebookTimeStamp() == 0)) {
        getVideosFromFacebook();
    }
    else if (userSignedInToTwitter() && (lastTweet().isEmpty())) {
        getVideosFromTwitter();
    }
}

void Sharing::getVideosFromFacebook() {
    setBusy(true);
    emit busy(tr("Retrieving videos from facebook"));
    QUrl url("https://graph.facebook.com/me/feed");

    if (facebookTimeStamp() > 0) {
        url.addQueryItem("since", QString::number(facebookTimeStamp()));
    }

    url.addQueryItem("limit", "100");
    url.addQueryItem("access_token", facebookToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFacebookVideos()));
}

void Sharing::checkFacebookVideos() {
    setBusy(false);
    emit busyProgressChanged(1);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit warning("Network error");
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 400) {
        setFacebookAccessToken(QString());
        emit signedOutOfFacebook();
        emit reauthenticateForFacebook();
        connect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(getVideosFromFacebook()));
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            emit warning("Error parsing server response");
        }
        else {
            setFacebookTimeStamp(QDateTime::currentMSecsSinceEpoch() / 1000);
            QVariantList results = result.value("data").toList();
            QStringList youtubeIds;
            QStringList dailymotionIds;
            QRegExp ytre("(youtube|youtu.be)", Qt::CaseInsensitive);
            QRegExp dmre("(dailymotion|dai.ly)", Qt::CaseInsensitive);

            for (int i = 0; i < results.size(); i++) {
                QString link = results.at(i).toMap().value("link").toString();

                if (!link.isEmpty()) {
                    if (link.contains(ytre)) {
                        youtubeIds.append(link.section(QRegExp("(v=|/)"), -1).left(11));
                    }
                    else if (link.contains(dmre)) {
                        dailymotionIds.append(link.section("/", -1).left(6));
                    }
                }
            }

            if (!youtubeIds.isEmpty()) {
                youtubeIds.removeDuplicates();
                emit gotYouTubeIds(youtubeIds);
            }
            if (!dailymotionIds.isEmpty()) {
                dailymotionIds.removeDuplicates();
                emit gotDailymotionIds(dailymotionIds);
            }
        }

        if (userSignedInToTwitter()) {
            getVideosFromTwitter();
        }
        else {
            resetVideoRefreshTime();
        }

        disconnect(this, SIGNAL(signedInToFacebook(QString)), this, SLOT(getVideosFromFacebook()));
    }

    reply->deleteLater();
}

void Sharing::getVideosFromTwitter() {
    setBusy(true);
    emit busy(tr("Retreiving videos from twitter"));
    QUrl url("http://api.twitter.com/1/statuses/home_timeline.json");
    url.addQueryItem("count", "200");
    url.addQueryItem("include_entities", "true");
    url.addQueryItem("trim_user", "true");
    url.addQueryItem("include_rts", "false");

    if (!lastTweet().isEmpty()) {
        url.addQueryItem("since_id", lastTweet());
    }

    QMap<QString, QString> params;
    params.insert("count", "200");
    params.insert("include_entities", "true");
    params.insert("trim_user", "true");
    params.insert("include_rts", "false");

    if (!lastTweet().isEmpty()) {
        params.insert("since_id", lastTweet());
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "cuteTube/1.3.1 (Nokia; Qt)");
    request.setRawHeader("Authorization", getTwitterOAuthHeader("GET", "http://api.twitter.com/1/statuses/home_timeline.json", params));
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkTwitterVideos()));
}

void Sharing::checkTwitterVideos() {
    setBusy(false);
    emit busyProgressChanged(1);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (userSignedInToTwitter())) {
        setTwitterAccount(QString(), QString());
        emit signedOutOfTwitter();
        emit reauthenticateForTwitter();
        connect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(getVideosFromTwitter()));
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantList results = Json::parse(response, ok).toList();

        if (!ok) {
            emit warning("Error parsing server response");
        }
        else {
            QStringList youtubeIds;
            QStringList dailymotionIds;
            QRegExp ytre("(youtube|youtu.be)", Qt::CaseInsensitive);
            QRegExp dmre("(dailymotion|dai.ly)", Qt::CaseInsensitive);

            if (!results.isEmpty()) {
                setLastTweet(results.first().toMap().value("id_str").toString());
            }

            while (!results.isEmpty()) {
                QVariantMap tweet = results.takeFirst().toMap();
                QVariantList urls = tweet.value("entities").toMap().value("urls").toList();

                while (!urls.isEmpty()) {
                    QString url = urls.takeFirst().toMap().value("expanded_url").toString();

                    if (!url.isEmpty()) {
                        if (url.contains(ytre)) {
                            youtubeIds.append(url.section(QRegExp("(v=|/)"), -1).left(11));
                        }
                        else if (url.contains(dmre)) {
                            dailymotionIds.append(url.section("/", -1).left(6));
                        }
                    }
                }
            }

            if (!youtubeIds.isEmpty()) {
                youtubeIds.removeDuplicates();
                emit gotYouTubeIds(youtubeIds);
            }
            if (!dailymotionIds.isEmpty()) {
                dailymotionIds.removeDuplicates();
                emit gotDailymotionIds(dailymotionIds);
            }
        }

        disconnect(this, SIGNAL(signedInToTwitter(QString,QString)), this, SLOT(getVideosFromTwitter()));
    }

    resetVideoRefreshTime();
    reply->deleteLater();
}

void Sharing::addVideosToCache(QList< QSharedPointer<VideoItem> > videos) {
    while (!videos.isEmpty()) {
        m_videoCache->insert(0, videos.takeFirst());
        emit videoAddedToCache(0);
    }

    emit videosAdded();
}
