#include "videocontrols.h"
#include "../base/utils.h"
#include "playbacklistdelegate.h"
#include "../base/urlgrabber.h"
#include "videoplayerbutton.h"
#include "../base/playbacklistmodel.h"
#include "../base/notifications.h"
#include <QSlider>
#include <QProgressBar>
#include <QLabel>
#include <QHBoxLayout>
#include <QTimer>
#include <QListView>

VideoControls::VideoControls(QMediaPlayer *player, QList< QSharedPointer<VideoItem> > playlist, QWidget *parent) :
    QWidget(parent),
    m_player(player),
    m_playlist(new PlaybackListModel(this)),
    m_listView(new QListView(this)),
    m_previousButton(new VideoPlayerButton(VideoPlayerButton::Previous, this)),
    m_playButton(new VideoPlayerButton(VideoPlayerButton::PlayPause, this)),
    m_nextButton(new VideoPlayerButton(VideoPlayerButton::Next, this)),
    m_backButton(new VideoPlayerButton(VideoPlayerButton::Back, this)),
    m_progressSlider(new QSlider(Qt::Horizontal, this)),
    m_bufferProgressBar(new QProgressBar(this)),
    m_positionLabel(new QLabel("00:00", this)),
    m_durationLabel(new QLabel("00:00", this)),
    m_hbox(new QHBoxLayout(this)),
    m_listTimer(new QTimer(this)),
    m_currentIndex(-1)
{
    this->setFixedSize(800, 70);
    this->setAttribute(Qt::WA_TranslucentBackground, true);
    this->setStyleSheet("background:transparent");

    m_listTimer->setSingleShot(true);
    m_listTimer->setInterval(3000);

    m_progressSlider->setFocusPolicy(Qt::NoFocus);
    m_previousButton->setFocusPolicy(Qt::NoFocus);
    m_playButton->setFocusPolicy(Qt::NoFocus);
    m_nextButton->setFocusPolicy(Qt::NoFocus);
    m_backButton->setFocusPolicy(Qt::NoFocus);
    m_listView->setFocusPolicy(Qt::NoFocus);

    m_playlist->addVideos(playlist);

    m_listView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_listView->setVerticalScrollMode(QListView::ScrollPerItem);
    m_listView->setModel(m_playlist);
    m_listView->setItemDelegate(new PlaybackListDelegate(m_listView));

    m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
    m_backButton->setIcon(QIcon::fromTheme("general_backspace"));

    m_positionLabel->setFixedSize(70, 70);
    m_durationLabel->setFixedSize(70, 70);
    m_positionLabel->setAlignment(Qt::AlignCenter);
    m_durationLabel->setAlignment(Qt::AlignCenter);

    QFont font;
    font.setPixelSize(18);

    m_positionLabel->setFont(font);
    m_durationLabel->setFont(font);

    m_bufferProgressBar->setTextVisible(true);
    m_bufferProgressBar->setFormat("Buffering");
    m_bufferProgressBar->setMinimum(0);
    m_bufferProgressBar->setMaximum(100);

    m_hbox->setContentsMargins(0, 0, 0, 0);
    m_hbox->setSpacing(0);
    m_hbox->addWidget(m_positionLabel, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_progressSlider, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_bufferProgressBar, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_listView, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_durationLabel, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_previousButton, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_playButton, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_nextButton, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_backButton, 0, Qt::AlignVCenter);

    m_bufferProgressBar->hide();
    m_listView->hide();

    this->connect(m_previousButton, SIGNAL(clicked()), this, SLOT(previous()));
    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(togglePlayPause()));
    this->connect(m_nextButton, SIGNAL(clicked()), this, SLOT(next()));
    this->connect(m_backButton, SIGNAL(clicked()), this, SIGNAL(back()));
    this->connect(m_progressSlider, SIGNAL(sliderReleased()), this, SLOT(setPosition()));
    this->connect(m_player, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onStateChanged(QMediaPlayer::State)));
    this->connect(m_player, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged(QMediaPlayer::MediaStatus)));
    this->connect(m_player, SIGNAL(bufferStatusChanged(int)), this, SLOT(onBufferStatusChanged(int)));
    this->connect(m_player, SIGNAL(bufferStatusChanged(int)), m_bufferProgressBar, SLOT(setValue(int)));
    this->connect(m_listTimer, SIGNAL(timeout()), this, SLOT(hideListView()));
    this->connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(UrlGrabber::instance(), SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), this, SLOT(onUrlReady(QUrl)));
    this->connect(UrlGrabber::instance(), SIGNAL(busyProgressChanged(int)), m_bufferProgressBar, SLOT(setValue(int)));
    this->connect(UrlGrabber::instance(), SIGNAL(busy(QString)), this, SLOT(onVideoLoading(QString)));

    this->next();
}

void VideoControls::onPositionChanged(qint64 position) {
    m_positionLabel->setText(Utils::durationFromMSecs(position));

    if (!m_progressSlider->isSliderDown()) {
        m_progressSlider->setValue(position / 1000);
    }
}

void VideoControls::onDurationChanged(qint64 duration) {
    m_durationLabel->setText(Utils::durationFromMSecs(duration));
    m_progressSlider->setMaximum(duration / 1000);
}

void VideoControls::onStateChanged(QMediaPlayer::State state) {
    switch (state) {
    case QMediaPlayer::PlayingState:
        m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Pause.png"));
        return;
    default:
        m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
        return;
    }
}

void VideoControls::onMediaStatusChanged(QMediaPlayer::MediaStatus status) {
    switch (status) {
    case QMediaPlayer::InvalidMedia:
        Notifications::showError(tr("Video cannot be played"));
        return;
    case QMediaPlayer::EndOfMedia:
        this->next();
        return;
    default:
        return;
    }
}

void VideoControls::onVideoLoading(const QString &message) {
    m_bufferProgressBar->setFormat(message);
    m_bufferProgressBar->show();
    m_progressSlider->hide();
    m_listView->hide();
    this->show();
}

void VideoControls::onUrlReady(const QUrl &url) {
    m_bufferProgressBar->setFormat(tr("Buffering"));
    m_player->setMedia(QMediaContent(url));
    m_player->play();
}

void VideoControls::onBufferStatusChanged(int progress) {
    this->setVisible(progress < 100);
    m_bufferProgressBar->setVisible(progress < 100);
    m_progressSlider->setVisible(progress == 100);
    m_listView->hide();
}

void VideoControls::onItemClicked(const QModelIndex &index) {
    m_currentIndex = index.row() - 1;
    this->next();
}

void VideoControls::togglePlayPause() {
    switch (m_player->state()) {
    case QMediaPlayer::PlayingState:
        m_player->pause();
        return;
    default:
        m_player->play();
        return;
    }
}

void VideoControls::previous() {
    if (m_currentIndex > 0) {
        m_currentIndex--;

        if (m_currentIndex < m_playlist->rowCount()) {
            m_listView->setCurrentIndex(m_playlist->index(m_currentIndex));
            QSharedPointer<VideoItem> video = m_playlist->get(m_currentIndex);

            switch (video.data()->service()) {
            case Services::NoService:
                m_player->setMedia(video.data()->url());
                m_player->play();
                return;
            default:
                UrlGrabber::instance()->getVideoUrl(video.data()->service(), video.data()->videoId());
                return;
            }
        }
    }
    else {
        m_player->setPosition(0);
        m_player->play();
    }
}

void VideoControls::next() {
    if (m_currentIndex < m_playlist->rowCount() - 1) {
        m_currentIndex++;
        m_listView->setCurrentIndex(m_playlist->index(m_currentIndex));
        QSharedPointer<VideoItem> video = m_playlist->get(m_currentIndex);

        switch (video.data()->service()) {
        case Services::NoService:
            m_player->setMedia(video.data()->url());
            m_player->play();
            return;
        default:
            UrlGrabber::instance()->getVideoUrl(video.data()->service(), video.data()->videoId());
            return;
        }
    }
    else {
        emit back();
    }
}

void VideoControls::keyPressEvent(QKeyEvent *event) {
    switch(event->key()) {
    case Qt::Key_Space:
        this->togglePlayPause();
        return;
    case Qt::Key_Left:
        this->previous();
        return;
    case Qt::Key_Right:
        this->next();
        return;
    case Qt::Key_Up:
        this->listViewUp();
        return;
    case Qt::Key_Down:
        this->listViewDown();
        return;
    case Qt::Key_Return:
        this->setPlaylistIndex();
        return;
    case Qt::Key_Enter:
        this->setPlaylistIndex();
        return;
    }
}

void VideoControls::setPosition() {
    qint64 pos = m_progressSlider->value() * 1000;
    m_player->setPosition(pos);
}

void VideoControls::hideListView() {
    m_listView->hide();
    m_listView->scrollTo(m_playlist->index(m_currentIndex));

    if (!m_bufferProgressBar->isVisible()) {
        m_progressSlider->show();
    }
}

void VideoControls::listViewUp() {
    if (!m_bufferProgressBar->isVisible()) {
        m_progressSlider->hide();
        m_listView->show();
        m_listView->scrollTo(m_listView->indexAt(QPoint(0, -5)));
        m_listTimer->start();
        this->show();
    }
}

void VideoControls::listViewDown() {
    if (!m_bufferProgressBar->isVisible()) {
        m_progressSlider->hide();
        m_listView->show();
        m_listView->scrollTo(m_listView->indexAt(QPoint(0, 75)));
        m_listTimer->start();
        this->show();
    }
}

void VideoControls::setPlaylistIndex() {
    if (m_listView->isVisible()) {
        m_currentIndex = m_listView->indexAt(QPoint(0, 0)).row() - 1;
        this->next();
    }
}

void VideoControls::showEvent(QShowEvent *event) {
    Q_UNUSED(event);

    this->onPositionChanged(m_player->position());
    this->onDurationChanged(m_player->duration());

    this->connect(m_player, SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    this->connect(m_player, SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
}

void VideoControls::hideEvent(QHideEvent *event) {
    Q_UNUSED(event);

    this->disconnect(m_player, SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    this->disconnect(m_player, SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
}
