#include "database.h"
#include "settings.h"
#include "utils.h"
#include <QFileInfo>
#include <QDir>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QRegExp>
#include <QDateTime>
#include <QUrl>
#include <QDebug>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif

Database* Database::self = 0;

Database::Database(QObject *parent) :
    QObject(parent),
    m_database(QSqlDatabase::addDatabase("QSQLITE"))
{
    if (!self) {
        self = this;
    }
#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    QDir dir;
    dir.mkpath("/home/user/cuteTube/");
    m_database.setDatabaseName("/home/user/cuteTube/cuteTube.db");
#elif (defined (SYMBIAN_OS))
    m_database.setDatabaseName("cuteTube.db");
#elif QT_VERSION >= 0x050000
    QDir dir(QStandardPaths::writableLocation(QStandardPaths::HomeLocation) + "/cuteTube/");
    dir.mkpath(dir.path());
    m_database.setDatabaseName(dir.path() + "/cuteTube.db");
#else
    QDir dir(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/cuteTube/");
    dir.mkpath(dir.path());
    m_database.setDatabaseName(dir.path() + "/cuteTube.db");
#endif
    this->initialize();
}

Database* Database::instance() {
    return !self ? new Database : self;
}

void Database::initialize() {
    if (m_database.open()) {
        m_database.exec("DROP TABLE IF EXISTS storedDownloads");
        m_database.exec("CREATE TABLE IF NOT EXISTS youtubeAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS dailymotionAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS vimeoAccounts (name TEXT UNIQUE, username TEXT, token TEXT, secret TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS facebookAccount (name TEXT UNIQUE, token TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS twitterAccount (name TEXT UNIQUE, token TEXT, secret TEXT)");
        m_database.close();
    }
    else {
        qWarning() << m_database.lastError().text();
    }
}

void Database::restoreAccounts() {
    this->restoreYouTubeAccount();
    this->restoreDailymotionAccount();
    this->restoreVimeoAccount();
    this->restoreFacebookAccount();
    this->restoreTwitterAccount();
}

bool Database::setYouTubeUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube account details"));

        return false;
    }
}

bool Database::setYouTubeAccessToken(const QString &token) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET token = ? WHERE active = 1");
        query.addBindValue(token);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube account details"));

        return false;
    }
}

void Database::restoreYouTubeAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString refresh;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, refresh FROM youtubeAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                refresh = query.value(2).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotYouTubeAccount(user, token, refresh);
        }
    }
}

bool Database::setDailymotionUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion account details"));

        return false;
    }
}

bool Database::setDailymotionAccessToken(const QString &token, const QString &refresh) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET token = ?, refresh = ? WHERE active = 1");
        query.addBindValue(token);
        query.addBindValue(refresh);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion account details"));

        return false;
    }
}

void Database::restoreDailymotionAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString refresh;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, refresh FROM dailymotionAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                refresh = query.value(2).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotDailymotionAccount(user, token, refresh);
        }
    }
}

bool Database::setVimeoUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo account details"));

        return false;
    }
}

bool Database::setVimeoAccessToken(const QString &token, const QString &secret) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET token = ?, secret = ? WHERE active = 1");
        query.addBindValue(token);
        query.addBindValue(secret);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo account details"));

        return false;
    }
}

void Database::restoreVimeoAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString secret;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, secret FROM vimeoAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                secret = query.value(2).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotVimeoAccount(user, token, secret);
        }
    }
}

bool Database::storeFacebookAccount(const QString &token) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO facebookAccount VALUES (?, ?)");
        query.addBindValue("facebook");
        query.addBindValue(token);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store facebook account details"));
        }
        else {
            emit gotFacebookAccount(token);
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store facebook account details"));

        return false;
    }
}

bool Database::deleteFacebookAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM facebookAccount WHERE name = ?");
        query.addBindValue("facebook");

        bool success = query.exec();

        if (success) {
            emit gotFacebookAccount(QString());
            emit info(tr("facebook account deleted. Please visit the facebook website to revoke access"));
        }
        else {
            emit error(tr("Database error. Unable to delete facebook account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to delete facebook account details"));

        return false;
    }
}

void Database::restoreFacebookAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString token;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT token FROM facebookAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                token = query.value(0).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotFacebookAccount(token);
        }
    }
}

bool Database::storeTwitterAccount(const QString &token, const QString &secret) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO twitterAccount VALUES (?, ?, ?)");
        query.addBindValue("twitter");
        query.addBindValue(token);
        query.addBindValue(secret);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store twitter account details"));
        }
        else {
            emit gotTwitterAccount(token, secret);
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store twitter account details"));

        return false;
    }
}

bool Database::deleteTwitterAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM twitterAccount WHERE name = ?");
        query.addBindValue("twitter");

        bool success = query.exec();

        if (success) {
            emit gotFacebookAccount(QString());
            emit info(tr("twitter account deleted. Please visit the twitter website to revoke access"));
        }
        else {
            emit error(tr("Database error. Unable to delete twitter account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to delete twitter account details"));

        return false;
    }
}

void Database::restoreTwitterAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString token;
        QString secret;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT token, secret FROM twitterAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                token = query.value(0).toString();
                secret = query.value(1).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotTwitterAccount(token, secret);
        }
    }
}
