#include "utils.h"
#include <QFileInfo>
#include <QFile>
#include <QRegExp>
#ifndef QML_USER_INTERFACE
#include <QPainter>
#include <QRect>
#include <QImage>
#endif

const QRegExp YOUTUBE_URL_RE("(http(s|)://(www.|m.|)youtube.com/(v/|.+)(v=|list=|)|http://youtu.be/)", Qt::CaseInsensitive);
const QRegExp DAILYMOTION_URL_RE("(http(s|)://(www.|)dailymotion.com/|http://dai.ly/)\\w+", Qt::CaseInsensitive);
const QRegExp VIMEO_URL_RE("http(s|)://vimeo.com/\\w+", Qt::CaseInsensitive);

Utils::Utils(QObject *parent) :
    QObject(parent)
{
}

Utils::~Utils() {}

QString Utils::durationFromSecs(int secs) {
    QTime time(0, 0);
    QString format = secs >= 3600 ? "hh:mm:ss" : "mm:ss";

    return time.addSecs(secs).toString(format);
}

QString Utils::durationFromMSecs(int msecs) {
    QTime time(0, 0);
    QString format = msecs >= 3600000 ? "hh:mm:ss" : "mm:ss";

    return time.addMSecs(msecs).toString(format);
}

QString Utils::fileSizeFromPath(const QString &filePath) {
    QFileInfo file(filePath);
    return Utils::fileSizeFromBytes(file.size());
}

QString Utils::fileSizeFromBytes(double bytes) {
    double kb = 1024;
    double mb = kb * 1024;
    double gb = mb * 1024;

    QString size;

    if (bytes > gb) {
        size = QString::number(bytes / gb, 'f', 2) + "GB";
    }
    else if (bytes > mb) {
        size = QString::number(bytes / mb, 'f', 2) + "MB";
    }
    else if (bytes > kb){
        size = QString::number(bytes / kb, 'f', 2) + "KB";
    }
    else {
        size = QString::number(bytes) + "B";
    }

    return size;
}

QString Utils::dateFromSecs(qint64 secs, bool showTime) {
    return Utils::dateFromMSecs(secs * 1000, showTime);
}

QString Utils::dateFromMSecs(qint64 msecs, bool showTime) {
    QString date;

    if (showTime) {
        date = QDateTime::fromMSecsSinceEpoch(msecs).toString("dd/MM/yyyy | HH:mm");
    }
    else {
        date = QDateTime::fromMSecsSinceEpoch(msecs).toString("dd/MM/yyyy");
    }

    return date;
}

QString Utils::toRichText(QString s) {
    QRegExp re("((http(s|)://|[^\\s<\"']+@)[^\\s<\"']+)");
    int pos = 0;

    while ((pos = re.indexIn(s, pos)) != -1) {
        QString link = re.cap(1);
        s.replace(pos, link.size(), QString("<a href='%1'>%2</a>").arg(link.contains('@') ? "mailto:" + link : link).arg(link));
        pos += re.matchedLength() * 2 + 15;        
    }

    return s.replace("\n", "<br>");
}

QVariantMap Utils::parseIdFromUrl(const QString &url) {
    QVariantMap result;

    if (YOUTUBE_URL_RE.indexIn(url) == 0) {
        result.insert("service", Services::YouTube);
        result.insert("id", url.section(QRegExp("v=|list=|/"), -1).section(QRegExp("&|\\?"), 0, 0));

        if ((url.contains("youtu.be")) || (url.contains("v=") || (url.contains("/v/")))) {
            result.insert("type", Queries::Videos);
        }
        else if (url.contains("list=")) {
            result.insert("type", Queries::Playlists);
        }
        else {
            result.insert("type", Queries::Users);
        }
    }
    else if (DAILYMOTION_URL_RE.indexIn(url) == 0) {
        result.insert("service", Services::Dailymotion);
        result.insert("id", url.section('/', -1).section('_', 0, 0));

        if ((url.contains("dai.ly") || (url.contains("/video/")))) {
            result.insert("type", Queries::Videos);
        }
        else if (url.contains("/playlist/")) {
            result.insert("type", Queries::Playlists);
        }
        else {
            result.insert("type", Queries::Users);
        }
    }
    else if (VIMEO_URL_RE.indexIn(url) == 0) {
        QString id = url.section('/', -1);

        result.insert("service", Services::Vimeo);
        result.insert("id", id);

        if (url.contains("/album/")) {
            result.insert("type", Queries::Playlists);
        }
        else if (id.contains(QRegExp("\\D+"))) {
            result.insert("type", Queries::Users);
        }
        else {
            result.insert("type", Queries::Videos);
        }
    }

    return result;
}

QString Utils::base64Id(const QString &artist, const QString &title) {
    return QString(artist.toUtf8().toBase64() + "-" + title.toUtf8().toBase64());
}

QString Utils::localDateTimeFromString(const QString &dateTimeString, Qt::DateFormat format) {
    QDateTime dt = QDateTime::fromString(dateTimeString, format);

    if (!dt.isValid()) {
        dt = QDateTime::currentDateTimeUtc();
    }

    return dt.toLocalTime().toString("dd/MM/yyyy | HH:mm");
}

QString Utils::unescape(const QString &s) {
    int unescapes = 0;
    QByteArray us = s.toUtf8();

    while ((us.contains('%')) && (unescapes < 10)) {
        us = QByteArray::fromPercentEncoding(us);
        unescapes++;
    }

    return QString(us);
}

void Utils::log(const QString &filePath, const QByteArray &message) {
    QFile lf(filePath);

    if (lf.open(QIODevice::Append)) {
        lf.write(QDateTime::currentDateTime().toString().toUtf8() + ": " + message + "\n");
    }

    lf.close();
}

#ifndef QML_USER_INTERFACE
void Utils::drawBorderImage(QPainter *painter, const QRect &rect, const QImage &image, int top, int right, int left, int bottom) {
    QRect imageRect = image.rect();

    // Top-left
    painter->drawImage(QRect(rect.left(), rect.top(), left, top),
                       image,
                       QRect(imageRect.left(), imageRect.top(), left, top));

    // Top-right
    painter->drawImage(QRect(rect.right() - right, rect.top(), right, top),
                       image,
                       QRect(imageRect.right() - right, imageRect.top(), right, top));

    // Bottom-left
    painter->drawImage(QRect(rect.left(), rect.bottom() - bottom, left, bottom),
                       image,
                       QRect(imageRect.left(), imageRect.bottom() - bottom, left, bottom));

    // Bottom-right
    painter->drawImage(QRect(rect.right() - right, rect.bottom() - bottom, bottom, right),
                       image,
                       QRect(imageRect.right() - right, imageRect.bottom() - bottom, right, bottom));

    // Top-middle
    painter->drawImage(QRect(rect.left() + left, rect.top(), rect.width() - (left + right), top),
                       image,
                       QRect(imageRect.left() + left, imageRect.top(), imageRect.width() - (left + right), top));

    // Bottom-middle
    painter->drawImage(QRect(rect.left() + left, rect.bottom() - bottom, rect.width() - (left + right), bottom),
                       image,
                       QRect(imageRect.left() + left, imageRect.bottom() - bottom, imageRect.width() - (left + right), bottom));

    // Left-middle
    painter->drawImage(QRect(rect.left(), rect.top() + top, left, rect.height() - (top + bottom)),
                       image,
                       QRect(imageRect.left(), imageRect.top() + top, left, imageRect.height() - (top + bottom)));

    // Right-middle
    painter->drawImage(QRect(rect.right() - right, rect.top() + top, right, rect.height() - (top + bottom)),
                       image,
                       QRect(imageRect.right() - right, imageRect.top() + top, right, imageRect.height() - (top + bottom)));

    // Center
    painter->drawImage(QRect(rect.left() + left, rect.top() + top, rect.width() - (left + right), rect.height() - (top + bottom)),
                       image,
                       QRect(imageRect.left() + left, imageRect.top() + top, imageRect.width() - (left + right), imageRect.height() - (top + bottom)));
}
#endif
