from PyQt4.QtCore import QRect, QPoint, Qt
from PyQt4.QtGui import QWidget, QPainter, QColor, QImage, QPen, QPixmap, QApplication, QPolygon
from threading import Thread
from warnings import warn
import sys
import math

from square import SquareGame, SquareObserver

colors = ["red", "blue", "green", "yellow", "navy"]
def get_color(player):
    if player < len(colors):
        return colors[player]
    else:
        warn("There aren't enough colors")
        return "cyan"


class AnimationManager():

    class Animation:
        def __init__(self, function, length, one_shot, default):
            self.running = False
            self.offset = 0
            self.length = length
            self.function = function
            self.one_shot = one_shot
            self.default = default

    def __init__(self):
        self.cur = 0
        self.animations = {}

    def tick(self):
        self.cur += 1

        for name, ani in self.animations.iteritems():
            if ani.running:
                return True

        return False

    def add_animation(self, name, function, length, one_shot = False, default = 1.0):
        ani = self.Animation(function, length, one_shot, default)
        self.animations[name] = ani
        
    def start_animation(self, name):
        ani = self.animations[name]
        ani.offset = self.cur
        ani.running = True

    def stop_animation(self, name):
        ani = self.animations[name]
        ani.running = False

    def get_value(self, name):
        ani = self.animations[name]

        if ani.running:
            cur = self.cur - ani.offset

            if ani.one_shot and cur >= ani.length:
                ani.running = False
            else:
                cur %= ani.length
                return ani.function(cur, ani.length)

        return ani.default

    def is_running(self, name):
        ani = self.animations[name]

        if ani.one_shot and self.cur - ani.offset >= ani.length:
            ani.running = False

        return ani.running


def ani_blink(cur, length):
    return 1 - (math.cos((cur * math.pi * 2) / length) + 1) / 2

def ani_fade_in(cur, length):
    return 1 - (math.cos((cur * math.pi) / length) + 1) / 2

class CuteGame(QWidget, SquareObserver):

    INNER_SPACE = 10

    def __init__(self, game, local_player, parent = None):
        QWidget.__init__(self, parent)

        self.setWindowTitle('CuteSquare')

        self.local_player = local_player

        self.game = game
        self.square = CuteSquare(game, local_player, self)
        self.cur_player = 0
        self.needs_update = True

        game.add_square_observer(self)

        self.setGeometry(150, 50, 600, 650)

    def point_set(self, player, point, squares):
        self.cur_player = (player + 1) % len(self.game.players)
        self.needs_update = True
        self.update()

    def resizeEvent(self, event):
        name_size = max(self.height() / 20, 30)

        width = self.width() - self.INNER_SPACE * 2
        height = self.height() - name_size - self.INNER_SPACE * 2

        size = min(width, height)

        x = (width - size) / 2 + self.INNER_SPACE
        y = (height - size) / 2 + self.INNER_SPACE

        self.square.setGeometry(x, y, size, size)

        self.square.resizeEvent(None)

        self.needs_update = True

    def paintEvent(self, event):
	print "Painting"	

        if self.needs_update:
            self.needs_update = False

            name_size = max(self.height() / 20, 30)

            paint = QPainter()
            paint.begin(self)

            font = paint.font()
            font.setPixelSize(name_size * 0.7)

            players = self.game.players
            step = self.width() / len(players)
            for i, p in enumerate(players):
                font.setBold(i == self.cur_player)
                font.setItalic(i not in self.local_player)

                paint.setFont(font)

                paint.setPen(QColor(get_color(i)))
                msg = "%s: %i" % (p.name, p.score)

                if self.cur_player == i:
                    msg = "> %s <" % msg

                rect = QRect(i*step, self.height() - name_size, step, name_size)

                paint.drawText(rect, Qt.AlignHCenter | Qt.AlignVCenter, msg)

            paint.end()


class CuteSquare(QWidget, SquareObserver):

    SLOT_SIZE = 0.8
    STONE_SIZE = 0.6
    TPS = 25

    def __init__(self, game, user, parent=None):
        QWidget.__init__(self, parent)

        self.game = game
        self.cur_player = 0

        self.user = user

        self.old_size = (0, 0)

        self.updates = []
        self.last = []

        self.new_squares = []
        self.selected = None
        self.won = None

        self.setWindowTitle('CuteSquare')

        game.add_square_observer(self)

        self.stone_img = QImage("stone.png")
        self.slot_img = QImage("slot.png")

        self.animan = AnimationManager()
        self.startTimer(1000/self.TPS)

        self.animan.add_animation('point_blink', ani_blink, self.TPS * 1)
        self.animan.add_animation('win_fade', ani_fade_in, self.TPS * 1, one_shot = True)
        self.animan.add_animation('square_fade', ani_fade_in, self.TPS * 1, one_shot = True)

    def point_set(self, player, point, squares):
        self.selected = None
        self.last.append(point)
        self.updates.append((player, point))
        self.cur_player = (player + 1) % len(self.game.players)
        self.new_squares.extend(((player, s) for s in squares))
        self.animan.start_animation('square_fade')
        self.update()

    def player_wins(self, player):
        self.cur_player = -1
        self.won = self.game.players[player]
        self.animan.start_animation('win_fade')

    def timerEvent(self, event):
        if self.animan.tick():
            # animations running
            self.update()

    def paintSlot(self, paint, point):
        xstep = self.width() / self.game.size
        ystep = self.height() / self.game.size

        w = xstep * self.SLOT_SIZE
        h = ystep * self.SLOT_SIZE
        xo = xstep * (1 - self.SLOT_SIZE) / 2
        yo = ystep * (1 - self.SLOT_SIZE) / 2

        xgo = ( self.width() - self.game.size * xstep ) / 2
        ygo = ( self.height() - self.game.size * ystep ) / 2

        paint.drawPixmap(xo + point[0]*xstep + xgo, yo + point[1]*ystep + ygo, self.slot_map)

    def paintStone(self, paint, point):
        xstep = self.width() / self.game.size
        ystep = self.height() / self.game.size

        w = xstep * self.STONE_SIZE
        h = ystep * self.STONE_SIZE
        xo = xstep * (1 - self.STONE_SIZE) / 2
        yo = ystep * (1 - self.STONE_SIZE) / 2

        xgo = ( self.width() - self.game.size * xstep ) / 2
        ygo = ( self.height() - self.game.size * ystep ) / 2

        paint.drawEllipse(xo + point[0]*xstep + xgo, yo + point[1]*ystep + ygo, w, h)
        paint.drawPixmap(xo + point[0]*xstep + xgo, yo + point[1]*ystep + ygo, self.stone_map)

    def paintCircle(self, paint, point, size):
        xstep = self.width() / self.game.size
        ystep = self.height() / self.game.size

        w = xstep * size
        h = ystep * size
        xo = xstep * (1 - size) / 2
        yo = ystep * (1 - size) / 2

        xgo = ( self.width() - self.game.size * xstep ) / 2
        ygo = ( self.height() - self.game.size * ystep ) / 2

        paint.drawEllipse(xo + point[0]*xstep + xgo, yo + point[1]*ystep + ygo, w, h)

    def paintSquare(self, paint, square):
        xstep = self.width() / self.game.size
        ystep = self.height() / self.game.size

        xgo = ( self.width() - self.game.size * xstep ) / 2 + xstep / 2
        ygo = ( self.height() - self.game.size * ystep ) / 2 + ystep / 2
        
        for i in range(4):
            a = square[i]
            b = square[(i + 1) % 4]

            paint.drawLine(a[0]*xstep+xgo, a[1]*ystep+ygo, b[0]*xstep+xgo, b[1]*ystep+ygo)

    def paintSurface(self, paint, square, player):
        xstep = self.width() / self.game.size
        ystep = self.height() / self.game.size

        xgo = ( self.width() - self.game.size * xstep ) / 2 + xstep / 2
        ygo = ( self.height() - self.game.size * ystep ) / 2 + ystep / 2

        ani = self.animan.get_value('square_fade')
        
        color = QColor(get_color(player))
        color.setAlpha(0x77 * ani)
        paint.setBrush(color)
        paint.setPen(color)

        # draw surface
        points = []
        for a in square:
            points.append(QPoint(a[0]*xstep+xgo, a[1]*ystep+ygo))
        paint.drawPolygon(QPolygon(points))

        color = QColor(0xFF, 0xFF, 0xFF, 0xFF * ani)
        paint.setPen(color)
        paint.setBrush(color)

        font = paint.font()
        font.setPixelSize(ystep / 2)
        paint.setFont(font)

        # draw points
        coords = zip(*square)
        msg = "+%i" % ((max(coords[0]) - min(coords[0]) + 1) ** 2)
        p1 = QPoint(min(coords[0])*xstep, min(coords[1])*ystep)
        p2 = QPoint(max(coords[0])*xstep, max(coords[1])*ystep)
        rect = QRect(p1, p2)
        rect.translate(xgo, ygo)

        paint.drawText(rect, Qt.AlignHCenter | Qt.AlignVCenter, msg)
        

    def paintPoints(self, paint):
        for i, player in enumerate(self.game.players):
            color = QColor(get_color(i))

            paint.setPen(QPen())
            paint.setBrush(color)

            for point in player.points:
                self.paintStone(paint, point)

    def paintSquares(self, paint):
        for i, player in enumerate(self.game.players):
            color = QColor(get_color(i))
            color.setAlpha(100)

            pen = QPen(color)
            pen.setWidth(min(self.width(), self.height())/150 + 1)
            paint.setPen(pen)

            for square in player.squares:
                if square not in (s[1] for s in self.new_squares):
                    self.paintSquare(paint, square)

    def paintBackground(self, paint):
        paint.setBrush(QColor("light gray"))
        paint.drawRect(0, 0, self.width(), self.height())

        paint.setPen(QPen())
        paint.setBrush(QColor("dark gray"))
        for x in range(self.game.size):
            for y in range(self.game.size):
                self.paintSlot(paint, (x, y))

    def paintEvent(self, event):
	print "Painting"	

        paint = QPainter()

        for player, point in self.updates:
            color = QColor(get_color(player))

            paint.begin(self.point_map)
            paint.setRenderHint(QPainter.Antialiasing)

            paint.setPen(QPen())
            paint.setBrush(color)

            self.paintStone(paint, point)

            paint.end()

        self.updates = []

        if self.new_squares:
            (player, square) = self.new_squares[0]

            if not self.animan.is_running('square_fade'):
                color = QColor(get_color(player))
                color.setAlpha(100)

                paint.begin(self.back_map)
                paint.setRenderHint(QPainter.Antialiasing);

                pen = QPen(color)
                pen.setWidth(min(self.width(), self.height())/150 + 1)
                paint.setPen(pen)

                self.paintSquare(paint, square)

                paint.end()

                del self.new_squares[0]
                if self.new_squares:
                    self.animan.start_animation('square_fade')

        paint.begin(self)
        paint.drawPixmap(0, 0, self.back_map)
        paint.drawPixmap(0, 0, self.point_map)
        paint.end()

        if self.new_squares:
            (player, square) = self.new_squares[0]

            if self.animan.is_running('square_fade'):
                paint.begin(self)
                paint.setRenderHint(QPainter.Antialiasing);

                self.paintSurface(paint, square, player)

                paint.end()

        while len(self.last) > len(self.game.players) - 1:
            del self.last[0]

        paint.begin(self)

        color = QColor(0xFF, 0xFF, 0xFF, 0x55)
        paint.setPen(color)
        paint.setBrush(color)
        for p in self.last:
            self.paintCircle(paint, p, self.STONE_SIZE)

        if self.selected:
            ani = self.animan.get_value('point_blink')
            color = QColor(get_color(self.cur_player))
            color.setAlpha(0x22 * ani + 0x22)
            paint.setPen(QPen(0))
            paint.setBrush(color)
            self.paintCircle(paint, self.selected, self.STONE_SIZE)

        if self.won:
            font = paint.font()
            font.setPixelSize(self.height() / 20)
            paint.setFont(font)

            flags = Qt.AlignHCenter | Qt.AlignVCenter
            rect = QRect(20, 20, self.width() - 40, self.height() - 40)

            msg = "%s wins with %i points" % (self.won.name, self.won.score)

            ani = self.animan.get_value('win_fade')

            paint.setPen(QColor(0x00, 0x00, 0x00, ani * 0xFF))
            paint.setBrush(QColor(0x00, 0x00, 0x00, ani * 0x88))
            bounding = paint.boundingRect(rect, flags, msg)

            bounding.adjust(-10, -10, 10, 10)

            paint.drawRect(bounding)

            paint.setPen(QColor(0xFF, 0xFF, 0xFF, ani * 0xFF))
            paint.drawText(rect, flags, msg)

        paint.end()

    def createMap(self):
        map = QPixmap(self.width(), self.height())
        map.fill(QColor(0, 0, 0, 0))
        return map

    def getItemMap(self, paint, image, size):
        w = self.width() / self.game.size * size
        h = self.height() / self.game.size * size

        map = QPixmap(w, h)
        map.fill(QColor(0, 0, 0, 0))
        paint.begin(map)
        paint.drawImage(QRect(0,0,w,h), image)
        paint.end()
        return map

    def resizeEvent(self, event):
        size = (self.width(), self.height())
        if self.old_size != size:
            self.old_size = size

            paint = QPainter()

            self.stone_map = self.getItemMap(paint, self.stone_img, self.STONE_SIZE)
            self.slot_map = self.getItemMap(paint, self.slot_img, self.SLOT_SIZE)

            self.back_map = self.createMap()
            paint.begin(self.back_map)
            paint.setRenderHint(QPainter.Antialiasing);
            self.paintBackground(paint)
            self.paintSquares(paint)
            paint.end()

            self.point_map = self.createMap()
            paint.begin(self.point_map)
            paint.setRenderHint(QPainter.Antialiasing);
            self.paintPoints(paint)
            paint.end()

    def mouseReleaseEvent (self, event):
        if self.new_squares:
            print "aborting animation"
            self.animan.stop_animation('square_fade')
        else:
            old = self.selected

            self.selected = None
            self.animan.stop_animation('point_blink')

            if self.cur_player in self.user:
                xstep = self.width() / self.game.size
                ystep = self.height() / self.game.size

                xgo = ( self.width() - self.game.size * xstep ) / 2
                ygo = ( self.height() - self.game.size * ystep ) / 2

                point = ((event.x() - xgo) / xstep, (event.y() - ygo) / ystep)

                if point[0] in range(self.game.size) and point[1] in range(self.game.size):
                    if self.game.point_free(point):
                        if old == point:
                            self.game.set_point(point, self.cur_player)
                        else:
                            self.animan.start_animation('point_blink')
                            self.selected = point
                    else:
                        print("Point already taken!")
                else:
                    print("Out of range!")
            else:
                print("It's not your turn!")

        self.update()


def start_qt(game, user = None):
    if user == None:
        user = range(len(game.players))

    app = QApplication(sys.argv)

    widget = CuteGame(game, user)
    widget.show()

    app.exec_()

