/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "tuneinpodcastlistmodel.h"
#include "../tunein/tunein.h"
#include "../tunein/tuneinpodcastlist.h"
#ifndef QML_USER_INTERFACE
#include "../thumbnailcache/thumbnailcache.h"
#endif

TuneInPodcastListModel::TuneInPodcastListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    #endif
    m_loading(false),
    m_queryType(Queries::Unknown)
{
    m_roleNames[ServiceRole] = "service";
    m_roleNames[IdRole] = "id";
    m_roleNames[TitleRole] = "title";
    m_roleNames[DescriptionRole] = "description";
    m_roleNames[LogoRole] = "logo";
    m_roleNames[GenreRole] = "genre";
    m_roleNames[CountryRole] = "country";
    m_roleNames[LanguageRole] = "language";
    m_roleNames[SourceRole] = "source";
    m_roleNames[FavouriteRole] = "favourite";
    m_roleNames[LastPlayedRole] = "lastPlayed";
    m_roleNames[SectionRole] = "section";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

TuneInPodcastListModel::~TuneInPodcastListModel() {
    this->clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> TuneInPodcastListModel::roleNames() const {
    return m_roleNames;
}
#endif

int TuneInPodcastListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant TuneInPodcastListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case ServiceRole:
        return m_list.at(index.row())->service();
    case IdRole:
        return m_list.at(index.row())->id();
    case TitleRole:
        return m_list.at(index.row())->title();
    case DescriptionRole:
        return m_list.at(index.row())->description();
    case LogoRole:
#ifndef QML_USER_INTERFACE
        return m_cache->thumbnail(m_list.at(index.row())->logo(), QSize(64, 64));
#else
        return m_list.at(index.row())->logo();
#endif
    case GenreRole:
        return m_list.at(index.row())->genre();
    case CountryRole:
        return m_list.at(index.row())->country();
    case LanguageRole:
        return m_list.at(index.row())->language();
    case SourceRole:
        return m_list.at(index.row())->source();
    case FavouriteRole:
        return m_list.at(index.row())->favourite();
    case LastPlayedRole:
        return m_list.at(index.row())->lastPlayed();
    case SectionRole:
        return m_list.at(index.row())->title().left(1).toUpper();
    default:
        return QVariant();
    }
}

QVariant TuneInPodcastListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool TuneInPodcastListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if ((!this->rowCount()) || (!index.isValid())) {
        return false;
    }

    switch (role) {
    case IdRole:
    case LastPlayedRole:
    case SectionRole:
        break;
    default:
        if (Station *podcast = this->get(index.row())) {
            return podcast->setProperty(this->roleNames().value(role), value);
        }
    }

    return false;
}

bool TuneInPodcastListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return this->setData(this->index(row), value, this->roleNames().key(role));
}

bool TuneInPodcastListModel::moreResults() const {
    return !m_nextUrl.isEmpty();
}

QUrl TuneInPodcastListModel::nextUrl() const {
    return m_nextUrl;
}

void TuneInPodcastListModel::setNextUrl(const QUrl &url) {
    m_nextUrl = url;
}

bool TuneInPodcastListModel::loading() const {
    return m_loading;
}

void TuneInPodcastListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QString TuneInPodcastListModel::searchQuery() const {
    return m_searchQuery;
}

void TuneInPodcastListModel::setSearchQuery(const QString &query) {
    if (query != this->searchQuery()) {
        m_searchQuery = query;
        emit searchQueryChanged(query);
    }
}

Queries::QueryType TuneInPodcastListModel::queryType() const {
    return m_queryType;
}

void TuneInPodcastListModel::setQueryType(Queries::QueryType type) {
    if (type != this->queryType()) {
        m_queryType = type;
        emit queryTypeChanged(type);
    }
}

Station* TuneInPodcastListModel::get(const QModelIndex &index) const {
    return this->get(index.row());
}

Station* TuneInPodcastListModel::get(int row) const {
    return (row >= 0) && (row < m_list.size()) ? m_list.at(row) : 0;
}

void TuneInPodcastListModel::searchPodcasts(const QString &query) {
    this->setLoading(true);
    this->setSearchQuery(query);
    this->setQueryType(Queries::PodcastSearch);

    TuneInPodcastList *list = TuneIn::searchPodcasts(query);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInPodcastList*)), this, SLOT(onPodcastListFinished(TuneInPodcastList*)));
    this->connect(list, SIGNAL(canceled(TuneInPodcastList*)), this, SLOT(onPodcastListCanceled(TuneInPodcastList*)));
}

void TuneInPodcastListModel::showPodcastsByGenre(const QString &id) {
    this->setLoading(true);
    this->setSearchQuery(id);
    this->setQueryType(Queries::PodcastGenres);

    TuneInPodcastList *list = TuneIn::getPodcastsByGenre(id);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInPodcastList*)), this, SLOT(onPodcastListFinished(TuneInPodcastList*)));
    this->connect(list, SIGNAL(canceled(TuneInPodcastList*)), this, SLOT(onPodcastListCanceled(TuneInPodcastList*)));
}

void TuneInPodcastListModel::showPodcastsByCountry(const QString &id) {
    this->setLoading(true);
    this->setSearchQuery(id);
    this->setQueryType(Queries::PodcastCountries);

    TuneInPodcastList *list = TuneIn::getPodcastsByCountry(id);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInPodcastList*)), this, SLOT(onPodcastListFinished(TuneInPodcastList*)));
    this->connect(list, SIGNAL(canceled(TuneInPodcastList*)), this, SLOT(onPodcastListCanceled(TuneInPodcastList*)));
}

void TuneInPodcastListModel::showPodcasts(const QString &url) {
    this->setLoading(true);
    this->setSearchQuery(url);
    this->setQueryType(Queries::Unknown);

    TuneInPodcastList *list = TuneIn::getPodcasts(url);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(TuneInPodcastList*)), this, SLOT(onPodcastListFinished(TuneInPodcastList*)));
    this->connect(list, SIGNAL(canceled(TuneInPodcastList*)), this, SLOT(onPodcastListCanceled(TuneInPodcastList*)));
}

void TuneInPodcastListModel::getMoreResults() {
    if ((this->moreResults()) && (!this->loading())) {
        this->setLoading(true);

        TuneInPodcastList *list = TuneIn::getPodcasts(this->nextUrl());
        list->setParent(this);
        this->connect(list, SIGNAL(finished(TuneInPodcastList*)), this, SLOT(onPodcastListFinished(TuneInPodcastList*)));
        this->connect(list, SIGNAL(canceled(TuneInPodcastList*)), this, SLOT(onPodcastListCanceled(TuneInPodcastList*)));
    }
}

void TuneInPodcastListModel::clear() {
    this->beginResetModel();
    qDeleteAll(m_list);
    m_list.clear();
    this->endResetModel();
}

void TuneInPodcastListModel::reload() {
    this->clear();

    switch (this->queryType()) {
    case Queries::PodcastSearch:
        this->searchPodcasts(this->searchQuery());
        return;
    case Queries::PodcastGenres:
        this->showPodcastsByGenre(this->searchQuery());
        return;
    case Queries::PodcastCountries:
        this->showPodcastsByCountry(this->searchQuery());
        return;
    default:
        this->showPodcasts(this->searchQuery());
        return;
    }
}

void TuneInPodcastListModel::onPodcastListFinished(TuneInPodcastList *list) {
    switch (list->error()) {
    case QNetworkReply::NoError:
        break;
    default:
        this->setLoading(false);
        emit error(list->errorString());
        list->deleteLater();
        return;
    }

    this->setNextUrl(list->nextUrl());
    this->addPodcasts(list->results());
    list->deleteLater();
}

void TuneInPodcastListModel::onPodcastListCanceled(TuneInPodcastList *list) {
    this->setLoading(false);
    list->deleteLater();
}

void TuneInPodcastListModel::addPodcasts(QList<Station *> podcasts) {
    if (podcasts.isEmpty()) {
        this->setLoading(false);
        return;
    }

    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount() + podcasts.size() - 1);
    m_list << podcasts;
    this->endInsertRows();
    emit countChanged(this->rowCount());

    foreach (Station *podcast, podcasts) {
        this->connect(podcast, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastUpdated(QString,QVariantMap)));
        this->connect(podcast, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastUpdated(QString,QString,QVariant)));
    }

    this->setLoading(false);
}

void TuneInPodcastListModel::addPodcast(Station *podcast) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(podcast);
    this->endInsertRows();
    emit countChanged(this->rowCount());

    this->connect(podcast, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastUpdated(QString,QVariantMap)));
    this->connect(podcast, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastUpdated(QString,QString,QVariant)));
}

void TuneInPodcastListModel::insertPodcast(int row, Station *podcast) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginInsertRows(QModelIndex(), row, row);
        m_list.insert(row, podcast);
        this->endInsertRows();
        emit countChanged(this->rowCount());

        this->connect(podcast, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastUpdated(QString,QVariantMap)));
        this->connect(podcast, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastUpdated(QString,QString,QVariant)));
    }
}

void TuneInPodcastListModel::removePodcast(int row) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row)->deleteLater();
        this->endRemoveRows();
        emit countChanged(this->rowCount());
    }
}

void TuneInPodcastListModel::removePodcast(int role, const QVariant &value) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removePodcast(indexes.first().row());
    }
}

void TuneInPodcastListModel::onPodcastUpdated(const QString &id, const QVariantMap &properties) {
    Q_UNUSED(properties)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        switch (this->queryType()) {
        case Queries::PodcastGenres:
            if (indexes.first().data(GenreRole).toString() != this->searchQuery()) {
                this->removePodcast(indexes.first().row());
                return;
            }

            break;
        case Queries::PodcastCountries:
            if (indexes.first().data(CountryRole).toString() != this->searchQuery()) {
                this->removePodcast(indexes.first().row());
                return;
            }

            break;
        default:
            break;
        }

        emit dataChanged(indexes.first(), indexes.first());
    }
}

void TuneInPodcastListModel::onPodcastUpdated(const QString &id, const QString &property, const QVariant &value) {
    Q_UNUSED(property)
    Q_UNUSED(value)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        switch (this->queryType()) {
        case Queries::PodcastGenres:
            if (indexes.first().data(GenreRole).toString() != this->searchQuery()) {
                this->removePodcast(indexes.first().row());
                return;
            }

            break;
        case Queries::PodcastCountries:
            if (indexes.first().data(CountryRole).toString() != this->searchQuery()) {
                this->removePodcast(indexes.first().row());
                return;
            }

            break;
        default:
            break;
        }

        emit dataChanged(indexes.first(), indexes.first());
    }
}

void TuneInPodcastListModel::onPodcastDeleted(const QString &id) {
    this->removePodcast(IdRole, id);
}

#ifndef QML_USER_INTERFACE
void TuneInPodcastListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif
