/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "tuneinpodcastfilterlistmodel.h"
#include "../base/settings.h"

TuneInPodcastFilterListModel::TuneInPodcastFilterListModel(QObject *parent) :
    QSortFilterProxyModel(parent),
    m_model(new TuneInPodcastListModel(this))
  #if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
  ,m_highlight(Settings::instance()->activeColor())
  #endif
{
    m_roleNames = m_model->roleNames();
    m_roleNames[QueryMatchRole] = "queryMatch";
    this->setSourceModel(m_model);
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->setFilterRole(TuneInPodcastListModel::TitleRole);
    this->setDynamicSortFilter(true);

    this->connect(m_model, SIGNAL(countChanged(int)), this, SIGNAL(countChanged(int)));
    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SIGNAL(loadingChanged(bool)));
    this->connect(m_model, SIGNAL(searchQueryChanged(QString)), this, SIGNAL(searchQueryChanged(QString)));
    this->connect(m_model, SIGNAL(queryTypeChanged(Queries::QueryType)), this, SIGNAL(queryTypeChanged(Queries::QueryType)));
}

TuneInPodcastFilterListModel::~TuneInPodcastFilterListModel() {}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> TuneInPodcastFilterListModel::roleNames() const {
    return m_roleNames;
}
#endif

QVariant TuneInPodcastFilterListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case QueryMatchRole:
        return this->queryMatch(QSortFilterProxyModel::data(index, this->filterRole()).toString());
    default:
        return QSortFilterProxyModel::data(index, role);
    }
}

QVariant TuneInPodcastFilterListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

bool TuneInPodcastFilterListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return QSortFilterProxyModel::setData(this->index(row, 0), value, this->roleNames().key(role));
}

bool TuneInPodcastFilterListModel::moreResults() const {
    return m_model->moreResults();
}

QUrl TuneInPodcastFilterListModel::nextUrl() const {
    return m_model->nextUrl();
}

bool TuneInPodcastFilterListModel::loading() const {
    return m_model->loading();
}

QString TuneInPodcastFilterListModel::searchQuery() const {
    return m_model->searchQuery();
}

Queries::QueryType TuneInPodcastFilterListModel::queryType() const {
    return m_model->queryType();
}

QString TuneInPodcastFilterListModel::query() const {
    return m_query;
}

void TuneInPodcastFilterListModel::setQuery(const QString &query) {
    if (query != this->query()) {
        m_query = query;
        this->setFilterRegExp(QRegExp(query, Qt::CaseInsensitive, QRegExp::FixedString));
        emit queryChanged(query);
    }
}

QString TuneInPodcastFilterListModel::queryMatch(QString str) const {
    QRegExp re(this->filterRegExp());

    if ((!re.isEmpty()) && (re.indexIn(str) > -1)) {
#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
        str.replace(re, "<u><font color=\"" + this->highlightColor() + "\">" + re.cap() + "</font></u>");
#else
        str.replace(re, "<b>" + re.cap() + "</b>");
#endif
    }

    return str;
}

#if (defined QML_USER_INTERFACE) && (!defined Q_WS_MAEMO_5)
QString TuneInPodcastFilterListModel::highlightColor() const {
    return m_highlight;
}

void TuneInPodcastFilterListModel::setHighlightColor(const QString &color) {
    if (color != this->highlightColor()) {
        m_highlight = color;
        emit highlightColorChanged(color);
    }
}
#endif

Station* TuneInPodcastFilterListModel::get(const QModelIndex &index) const {
    return m_model->get(this->mapToSource(index));
}

Station* TuneInPodcastFilterListModel::get(int row) const {
    return m_model->get(this->mapToSource(this->index(row, 0)).row());
}

void TuneInPodcastFilterListModel::searchPodcasts(const QString &query) {
    m_model->searchPodcasts(query);
}

void TuneInPodcastFilterListModel::showPodcastsByGenre(const QString &id) {
    m_model->showPodcastsByGenre(id);
}

void TuneInPodcastFilterListModel::showPodcastsByCountry(const QString &id) {
    m_model->showPodcastsByCountry(id);
}

void TuneInPodcastFilterListModel::showPodcasts(const QString &url) {
    m_model->showPodcasts(url);
}

void TuneInPodcastFilterListModel::getMoreResults() {
    m_model->getMoreResults();
}

void TuneInPodcastFilterListModel::clear() {
    m_model->clear();
}

void TuneInPodcastFilterListModel::reload() {
    m_model->reload();
}
