/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QThread>
#include <QImage>
#include <QUrl>
#include <QNetworkReply>
#include <QCoreApplication>
#include <qcoreevent.h>
#include "thumbnailcache.h"
#include "thumbnaildownloader.h"

#if (QT_VERSION < QT_VERSION_CHECK(4, 7, 0))
uint qHash(const QUrl &url) { return qHash(url.toString()); }
#endif

class Thumbnail
{

public:
    Thumbnail(const QUrl &url)
        : url(url)
    {
    }

    const QUrl url;
    QImage image;
};

class ThumbnailEvent : public QEvent
{

public:
    ThumbnailEvent(const QUrl &url, const QImage &image)
        : QEvent(QEvent::User)
        , url(url)
        , image(image)
    {
    }

    const QUrl url;
    const QImage image;
};

ThumbnailCache* ThumbnailCache::self = 0;
QThread* ThumbnailCache::cacheThread = 0;
QCache<QUrl, Thumbnail> ThumbnailCache::thumbnails;

ThumbnailCache::ThumbnailCache(QObject *parent) :
    QObject(parent)
{
    if (!self) {
        self = this;
        cacheThread = new QThread(this);
        cacheThread->start();
        this->connect(this, SIGNAL(destroyed()), cacheThread, SLOT(quit()));
    }

    this->moveToThread(cacheThread);

    ThumbnailDownloader *downloader = new ThumbnailDownloader;
    this->connect(this, SIGNAL(downloadThumbnail(QUrl)), downloader, SLOT(downloadThumbnail(QUrl)));
    this->connect(downloader, SIGNAL(gotThumbnail(QNetworkReply*)), this, SLOT(onThumbnailDownloaded(QNetworkReply*)));
    this->connect(this, SIGNAL(destroyed()), downloader, SLOT(deleteLater()));
}

ThumbnailCache::~ThumbnailCache() {}

ThumbnailCache* ThumbnailCache::instance() {
    return !self ? new ThumbnailCache : self;
}

QImage ThumbnailCache::thumbnail(const QUrl &url, const QSize &scale) {
    if (Thumbnail *thumbnail = thumbnails.object(url)) {
        return (!scale.isEmpty()) && (!thumbnail->image.isNull())
                ? thumbnail->image.scaled(scale, Qt::IgnoreAspectRatio, Qt::SmoothTransformation)
                : thumbnail->image;
    }
    else {
        m_urls.enqueue(url);
        thumbnails.insert(url, new Thumbnail(url));

        if (m_urls.size() == 1) {
            emit downloadThumbnail(url);
        }

        return QImage();
    }
}

void ThumbnailCache::nextThumbnail() {
    if (!m_urls.isEmpty()) {
        emit downloadThumbnail(m_urls.dequeue());
    }
}

bool ThumbnailCache::event(QEvent *event) {
    if (event->type() == QEvent::User) {
        ThumbnailEvent *thumbnailEvent = static_cast<ThumbnailEvent *>(event);

        if (Thumbnail *thumbnail = thumbnails.object(thumbnailEvent->url)) {
            thumbnail->image = thumbnailEvent->image;

            emit thumbnailReady();

            nextThumbnail();
        }

        return true;
    }
    else {
        return QObject::event(event);
    }
}

void ThumbnailCache::onThumbnailDownloaded(QNetworkReply *reply) {
    QUrl url = reply->request().url();
    QImage image;
    image.loadFromData(reply->readAll());
    reply->deleteLater();
    QCoreApplication::postEvent(this, new ThumbnailEvent(url, image));
}
