/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLineEdit>
#include <QGridLayout>
#include <QRegExpValidator>
#include "tuneinwindow.h"
#include "listview.h"
#include "valueselector.h"
#include "homescreenlistdelegate.h"
#include "tuneinbrowselistwindow.h"
#include "tuneinstationlistwindow.h"
#include "tuneinpodcastlistwindow.h"
#include "nowplayingbutton.h"
#include "../tunein/tuneinhomescreenmodel.h"
#include "../base/searchlistmodel.h"
#include "../base/selectionmodels.h"
#include "../base/settings.h"

TuneInWindow::TuneInWindow(QWidget *parent) :
    StackedWindow(parent),
    m_model(new TuneInHomescreenModel(this)),
    m_searchModel(new SearchListModel(this)),
    m_searchEdit(new QLineEdit(this)),
    m_searchTypeSelector(new ValueSelector(tr("Search for"), this)),
    m_view(new ListView(this)),
    m_nowPlayingButton(new NowPlayingButton(this))
{
    this->setWindowTitle(tr("TuneIn"));
    this->setCentralWidget(new QWidget(this));

    m_searchEdit->setPlaceholderText(tr("Search"));
    m_searchEdit->setValidator(new QRegExpValidator(QRegExp("^\\S.+"), m_searchEdit));
    m_searchEdit->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    m_searchTypeSelector->setModel(new StationTypeModel(m_searchTypeSelector));
    m_searchTypeSelector->setValue(Settings::instance()->searchType());

    m_view->setModel(m_model);
    m_view->setItemDelegate(new HomescreenListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_grid = new QGridLayout(this->centralWidget());
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_searchEdit, 0, 0);
    m_grid->addWidget(m_searchTypeSelector, 0, 1);
    m_grid->addWidget(m_view, 1, 0, 1, 2);

    this->connect(m_searchEdit, SIGNAL(returnPressed()), this, SLOT(search()));
    this->connect(m_searchTypeSelector, SIGNAL(valueChanged(QVariant)), this, SLOT(onSearchTypeChanged(QVariant)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
}

void TuneInWindow::onItemClicked(const QModelIndex &index) {
    switch (index.row()) {
    case 0:
        this->showGenres();
        return;
    case 1:
        this->showCountries();
        return;
    case 2:
        this->showPodcastGenres();
        return;
    default:
        return;
    }
}

void TuneInWindow::onSearchTypeChanged(const QVariant &type) {
    Settings::instance()->setSearchType(static_cast<Stations::StationType>(type.toInt()));
}

void TuneInWindow::search() {
    switch (m_searchTypeSelector->currentValue().toInt()) {
    case Stations::Podcast:
        this->searchPodcasts();
        return;
    default:
        this->searchStations();
        return;
    }
}

void TuneInWindow::searchStations() {
    TuneInStationListWindow *window = new TuneInStationListWindow(this);
    window->searchStations(m_searchEdit->text());
    window->show();
    m_searchModel->addSearch(m_searchEdit->text());
    m_searchEdit->clear();
}

void TuneInWindow::searchPodcasts() {
    TuneInPodcastListWindow *window = new TuneInPodcastListWindow(this);
    window->searchPodcasts(m_searchEdit->text());
    window->show();
    m_searchModel->addSearch(m_searchEdit->text());
    m_searchEdit->clear();
}

void TuneInWindow::showGenres() {
    TuneInBrowseListWindow *window = new TuneInBrowseListWindow(this);
    window->showGenres();
    window->show();
}

void TuneInWindow::showCountries() {
    TuneInBrowseListWindow *window = new TuneInBrowseListWindow(this);
    window->showCountries();
    window->show();
}

void TuneInWindow::showPodcastGenres() {
    TuneInBrowseListWindow *window = new TuneInBrowseListWindow(this);
    window->showGenres();
    window->show();
}
