/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLabel>
#include <QVBoxLayout>
#include <QMaemo5Style>
#include "tuneinbrowselistwindow.h"
#include "tuneinstationlistwindow.h"
#include "tuneinpodcastlistwindow.h"
#include "tuneinbrowselistdelegate.h"
#include "listview.h"
#include "filterbox.h"
#include "nowplayingbutton.h"
#include "../tunein/tuneinpairlistmodel.h"

TuneInBrowseListWindow::TuneInBrowseListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new TuneInPairListModel(this)),
    m_filterBox(new FilterBox(this)),
    m_noResultsLabel(new QLabel(this))
{
    this->setCentralWidget(m_noResultsLabel);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TuneInBrowseListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_filterBox->hide();

    m_vbox = new QVBoxLayout(m_noResultsLabel);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), this, SLOT(onScrollingStopped()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(cleared()), NowPlayingButton::instance(), SLOT(poke()));
}

void TuneInBrowseListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void TuneInBrowseListWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (m_model->rowCount() > 0) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        NowPlayingButton::instance()->hide();
        m_filterBox->show();
        m_filterBox->setFocus(Qt::OtherFocusReason);
        m_filterBox->setFilterText(event->text());
    }
}

void TuneInBrowseListWindow::showGenres() {
    this->setWindowTitle(tr("Stations by genre"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No genres found")));
    m_model->showGenres();
}

void TuneInBrowseListWindow::showPodcastGenres() {
    this->setWindowTitle(tr("Podcasts by genre"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No genres found")));
    m_model->showPodcastGenres();
}

void TuneInBrowseListWindow::showCountries() {
    this->setWindowTitle(tr("Stations by country"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No countries found")));
    m_model->showCountries();
}

void TuneInBrowseListWindow::showLanguages() {
    this->setWindowTitle(tr("Stations by language"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No languages found")));
    m_model->showLanguages();
}

void TuneInBrowseListWindow::showNames(const QString &id, const QString &filter) {
    this->setWindowTitle(tr("Stations by name"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No stations found")));
    m_model->showNames(id, filter);
}

void TuneInBrowseListWindow::showPodcastNames(const QString &id, const QString &filter) {
    this->setWindowTitle(tr("Podcasts by name"));
    m_noResultsLabel->setText(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No podcasts found")));
    m_model->showPodcastNames(id, filter);
}

void TuneInBrowseListWindow::clear() {
    m_model->clear();
}

void TuneInBrowseListWindow::reload() {
    m_model->reload();
}

void TuneInBrowseListWindow::onScrollingStopped() {
    if (m_filterBox->isHidden()) {
        NowPlayingButton::instance()->poke();
    }
}

void TuneInBrowseListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_view->setHidden((!loading) && (m_model->rowCount() == 0));
}

void TuneInBrowseListWindow::onCountChanged(int count) {
    m_view->setHidden((count == 0) && (!m_model->loading()));
}

void TuneInBrowseListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        switch (m_model->queryType()) {
        case Queries::Names:
        {
            TuneInStationListWindow *window = new TuneInStationListWindow(this);
            window->showStations(index.data(TuneInPairListModel::NameRole).toString(), index.data(TuneInPairListModel::ValueRole).toString());
            window->show();
            break;
        }
        case Queries::PodcastNames:
        {
            TuneInPodcastListWindow *window = new TuneInPodcastListWindow(this);
            window->showPodcasts(index.data(TuneInPairListModel::NameRole).toString(), index.data(TuneInPairListModel::ValueRole).toString());
            window->show();
            break;
        }
        case Queries::Genres:
        {
            TuneInBrowseListWindow *window = new TuneInBrowseListWindow(this);
            window->showNames(index.data(TuneInPairListModel::ValueRole).toString(), "s");
            window->show();
            break;
        }
        case Queries::PodcastGenres:
        {
            TuneInBrowseListWindow *window = new TuneInBrowseListWindow(this);
            window->showPodcastNames(index.data(TuneInPairListModel::ValueRole).toString(), "p");
            window->show();
            break;
        }
        default:
        {
            TuneInBrowseListWindow *window = new TuneInBrowseListWindow(this);
            window->showNames(index.data(TuneInPairListModel::ValueRole).toString(), "s:country");
            window->show();
            break;
        }
        }
    }
}
