/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QGridLayout>
#include <QLabel>
#include <QPushButton>
#include <QSlider>
#include "nowplayingwindow.h"
#include "separatorlabel.h"
#include "logolabel.h"
#include "scrollarea.h"
#include "screen.h"
#include "../base/utils.h"
#include "../thumbnailcache/thumbnailcache.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

NowPlayingWindow::NowPlayingWindow(QWidget *parent) :
    StackedWindow(parent),
    m_descriptionLabel(new QLabel(this)),
    m_titleLabel(new QLabel(this)),
    m_artistLabel(new QLabel(this)),
    m_albumLabel(new QLabel(this)),
    m_genreLabel(new QLabel(this)),
    m_bitrateLabel(new QLabel(this)),
    m_positionLabel(new QLabel("00:00", this)),
    m_durationLabel(new QLabel(this)),
    m_statusLabel(new QLabel(this)),
    m_logoLabel(new LogoLabel(this)),
    m_scrollArea(new ScrollArea(this)),
    m_controls(new QWidget(this)),
    m_playButton(new QPushButton(this)),
    m_sleepTimerButton(new QPushButton(QIcon::fromTheme("general_clock"), "", this)),
    m_slider(new QSlider(Qt::Horizontal, this))
{
    this->setCentralWidget(new QWidget(this));

    m_grid = new QGridLayout(this->centralWidget());

    m_descriptionLabel->setWordWrap(true);
    m_titleLabel->setWordWrap(true);
    m_artistLabel->setWordWrap(true);
    m_albumLabel->setWordWrap(true);
    m_genreLabel->setWordWrap(true);
    m_statusLabel->setWordWrap(true);

    m_positionLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
    m_durationLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    m_logoLabel->setFixedSize(124, 124);
    m_logoLabel->setInteractive(false);

    m_playButton->setFixedHeight(100);
    m_playButton->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    m_sleepTimerButton->setFixedHeight(100);
    m_sleepTimerButton->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    m_sleepTimerButton->setCheckable(true);
    m_sleepTimerButton->setChecked(MediaPlayer::instance()->sleepTimerEnabled());

    m_slider->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);
    vbox->setContentsMargins(0, 0, 0, 0);
    vbox->addWidget(m_logoLabel);
    vbox->addWidget(m_descriptionLabel);
    vbox->addWidget(new SeparatorLabel(tr("Now playing"), this));
    vbox->addWidget(m_titleLabel);
    vbox->addWidget(m_artistLabel);
    vbox->addWidget(m_albumLabel);
    vbox->addWidget(m_genreLabel);
    vbox->addWidget(m_bitrateLabel);
    vbox->addSpacerItem(new QSpacerItem(10, 10, QSizePolicy::Expanding, QSizePolicy::Expanding));

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    QGridLayout *grid = new QGridLayout(m_controls);
    grid->setContentsMargins(0, 0, 0, 0);
    grid->addWidget(m_playButton, 0, 0);
    grid->addWidget(m_sleepTimerButton, 0, 1);
    grid->addWidget(m_slider, 1, 0, 1, 2);
    grid->addWidget(m_positionLabel, 2, 0);
    grid->addWidget(m_durationLabel, 2, 1);
    grid->addWidget(m_statusLabel, 3, 0, 1, 2);

    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(onPlayButtonClicked()));
    this->connect(m_sleepTimerButton, SIGNAL(clicked(bool)), MediaPlayer::instance(), SLOT(setSleepTimerEnabled(bool)));
    this->connect(m_slider, SIGNAL(sliderReleased()), this, SLOT(onSliderReleased()));
    this->connect(Screen::instance(), SIGNAL(screenLockStateChanged(bool)), this, SLOT(onScreenLockStateChanged(bool)));

    this->onOrientationChanged();
}

void NowPlayingWindow::showEvent(QShowEvent *event) {
    this->connectPlaybackSignals();
    StackedWindow::showEvent(event);
}

void NowPlayingWindow::hideEvent(QHideEvent *event) {
    this->disconnectPlaybackSignals();
    StackedWindow::hideEvent(event);
}

void NowPlayingWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(400);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_controls, 0, 1, Qt::AlignVCenter);
    }
    else {
        m_grid->removeWidget(m_controls);
        m_grid->addWidget(m_controls, 0, 1, Qt::AlignVCenter);
    }
}

void NowPlayingWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_controls, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_controls);
        m_grid->addWidget(m_controls, 1, 0, Qt::AlignBottom);
    }
}

void NowPlayingWindow::connectPlaybackSignals() {
    MediaPlayer::instance()->setTickInterval(1000);
    this->connect(MediaPlayer::instance(), SIGNAL(currentStationChanged(Station*)), this, SLOT(onCurrentStationChanged(Station*)), Qt::UniqueConnection);
    this->connect(MediaPlayer::instance(), SIGNAL(metaDataChanged()), this, SLOT(onMetaDataChanged()), Qt::UniqueConnection);
    this->connect(MediaPlayer::instance(), SIGNAL(stateChanged(MediaState::State)), this, SLOT(onPlaybackStateChanged(MediaState::State)), Qt::UniqueConnection);
    this->connect(MediaPlayer::instance(), SIGNAL(positionChanged(qint64)), this, SLOT(onPlaybackPositionChanged(qint64)), Qt::UniqueConnection);
    this->connect(MediaPlayer::instance(), SIGNAL(durationChanged(qint64)), this, SLOT(onPlaybackDurationChanged(qint64)), Qt::UniqueConnection);
    this->connect(MediaPlayer::instance(), SIGNAL(sleepTimerEnabledChanged(bool)), this, SLOT(onSleepTimerEnabledChanged(bool)), Qt::UniqueConnection);
    this->connect(MediaPlayer::instance(), SIGNAL(sleepTimerRemainingChanged(qint64)), this, SLOT(onSleepTimerRemainingChanged(qint64)), Qt::UniqueConnection);
    this->onCurrentStationChanged(MediaPlayer::instance()->currentStation());
    this->onMetaDataChanged();
    this->onPlaybackStateChanged(MediaPlayer::instance()->state());
    this->onPlaybackDurationChanged(MediaPlayer::instance()->duration());
    this->onPlaybackPositionChanged(MediaPlayer::instance()->position());
    this->onSleepTimerEnabledChanged(MediaPlayer::instance()->sleepTimerEnabled());
}

void NowPlayingWindow::disconnectPlaybackSignals() {
    MediaPlayer::instance()->setTickInterval(0);
    this->disconnect(MediaPlayer::instance(), 0, this, 0);
}

void NowPlayingWindow::onCurrentStationChanged(Station *station) {
    if (station) {
        this->setWindowTitle(station->title());
        this->setLogo();
    }
}

void NowPlayingWindow::onPlayButtonClicked() {
    switch (MediaPlayer::instance()->state()) {
    case MediaState::Stopped:
    case MediaState::Paused:
    case MediaState::Error:
        MediaPlayer::instance()->play();
        return;
    default:
        if (MediaPlayer::instance()->duration() > 0) {
            MediaPlayer::instance()->pause();
        }
        else {
            MediaPlayer::instance()->stop();
        }

        return;
    }
}

void NowPlayingWindow::onSliderReleased() {
    MediaPlayer::instance()->setPosition(m_slider->value());
}

void NowPlayingWindow::onMetaDataChanged() {
    if (Station *station = MediaPlayer::instance()->currentStation()) {
        m_descriptionLabel->setText(MediaPlayer::instance()->description().isEmpty() ? station->description() : MediaPlayer::instance()->description());
        m_titleLabel->setText(QString("%1: %2").arg(tr("Title")).arg(MediaPlayer::instance()->title().isEmpty() ? station->title() : MediaPlayer::instance()->title()));
        m_artistLabel->setText(QString("%1: %2").arg(tr("Artist")).arg(MediaPlayer::instance()->artist().isEmpty() ? tr("Unknown") : MediaPlayer::instance()->artist()));
        m_albumLabel->setText(QString("%1: %2").arg(tr("Album")).arg(MediaPlayer::instance()->album().isEmpty() ? tr("Unknown") : MediaPlayer::instance()->album()));
        m_genreLabel->setText(QString("%1: %2").arg(tr("Genre")).arg(MediaPlayer::instance()->genre().isEmpty() ? station->genre() : MediaPlayer::instance()->genre()));
        m_bitrateLabel->setText(QString("%1: %2").arg(tr("Bitrate")).arg(!MediaPlayer::instance()->bitRate() ? tr("Unknown") : Utils::fileSizeFromBytes(MediaPlayer::instance()->bitRate()) + "/s"));
    }
}

void NowPlayingWindow::onPlaybackStateChanged(MediaState::State state) {
    m_statusLabel->setText(QString("%1: %2").arg(tr("Status")).arg(MediaPlayer::instance()->stateString()));

    switch (state) {
    case MediaState::Stopped:
    case MediaState::Paused:
    case MediaState::Error:
        m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
        return;
    default:
        m_playButton->setIcon(MediaPlayer::instance()->duration() > 0 ? QIcon("/etc/hildon/theme/mediaplayer/Pause.png")
                                                          : QIcon("/etc/hildon/theme/mediaplayer/Stop.png"));
        return;
    }
}

void NowPlayingWindow::onPlaybackPositionChanged(qint64 position) {
    m_positionLabel->setText(Utils::durationFromMSecs(position));

    if ((m_slider->isEnabled()) && (!m_slider->isSliderDown())) {
        m_slider->setValue(position);
    }
}

void NowPlayingWindow::onPlaybackDurationChanged(qint64 duration) {
    if (duration > 0) {
        m_durationLabel->setText(Utils::durationFromMSecs(duration));
        m_slider->setEnabled(true);
        m_slider->setMaximum(duration);
    }
    else {
        m_durationLabel->clear();
        m_slider->setEnabled(false);
    }
}

void NowPlayingWindow::onSleepTimerEnabledChanged(bool enabled) {
    m_sleepTimerButton->setText(enabled ? Utils::durationFromMSecs(MediaPlayer::instance()->sleepTimerRemaining()) : "");
}

void NowPlayingWindow::onSleepTimerRemainingChanged(qint64 remaining) {
    m_sleepTimerButton->setText(MediaPlayer::instance()->sleepTimerEnabled() ? Utils::durationFromMSecs(remaining) : "");
}

void NowPlayingWindow::onScreenLockStateChanged(bool locked) {
    if (locked) {
        this->disconnectPlaybackSignals();
    }
    else if (this->isVisible()) {
        this->connectPlaybackSignals();
    }
}

void NowPlayingWindow::setLogo() {
    if (Station *station = MediaPlayer::instance()->currentStation()) {
        QImage logo = ThumbnailCache::instance()->thumbnail(station->logo(), QSize(124, 124));

        if (logo.isNull()) {
            this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setLogo()), Qt::UniqueConnection);
        }
        else {
            this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setLogo()));
        }

        if (logo.isNull()) {
            logo = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_stream.png");
        }

        m_logoLabel->setImage(logo);
    }
}
