/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "podcastlistmodel.h"
#include "database.h"

PodcastListModel::PodcastListModel(QObject *parent) :
    QAbstractListModel(parent),
    m_loading(false),
    m_queryType(Queries::Unknown)
{
    m_roleNames[ServiceRole] = "service";
    m_roleNames[IdRole] = "id";
    m_roleNames[TitleRole] = "title";
    m_roleNames[DescriptionRole] = "description";
    m_roleNames[LogoRole] = "logo";
    m_roleNames[GenreRole] = "genre";
    m_roleNames[CountryRole] = "country";
    m_roleNames[LanguageRole] = "language";
    m_roleNames[SourceRole] = "source";
    m_roleNames[FavouriteRole] = "favourite";
    m_roleNames[LastPlayedRole] = "lastPlayed";
    m_roleNames[SectionRole] = "section";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif

    this->connect(Database::instance(), SIGNAL(podcastAdded()), this, SLOT(reload()), Qt::UniqueConnection);
    this->connect(Database::instance(), SIGNAL(podcastDeleted(QString)), this, SLOT(onPodcastDeleted(QString)), Qt::UniqueConnection);
}

PodcastListModel::~PodcastListModel() {
    this->clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> PodcastListModel::roleNames() const {
    return m_roleNames;
}
#endif

int PodcastListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant PodcastListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case ServiceRole:
        return m_list.at(index.row())->service();
    case IdRole:
        return m_list.at(index.row())->id();
    case TitleRole:
        return m_list.at(index.row())->title();
    case DescriptionRole:
        return m_list.at(index.row())->description();
    case LogoRole:
        return m_list.at(index.row())->logo();
    case GenreRole:
        return m_list.at(index.row())->genre();
    case CountryRole:
        return m_list.at(index.row())->country();
    case LanguageRole:
        return m_list.at(index.row())->language();
    case SourceRole:
        return m_list.at(index.row())->source();
    case FavouriteRole:
        return m_list.at(index.row())->favourite();
    case LastPlayedRole:
        return m_list.at(index.row())->lastPlayed();
    case SectionRole:
        switch (this->queryType()) {
        case Queries::RecentlyPlayedStations:
            return m_list.at(index.row())->lastPlayedString();
        default:
            return m_list.at(index.row())->title().left(1).toUpper();
        }
    default:
        return QVariant();
    }
}

QVariant PodcastListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool PodcastListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if ((!this->rowCount()) || (!index.isValid())) {
        return false;
    }

    switch (role) {
    case IdRole:
    case LastPlayedRole:
    case SectionRole:
        return false;
    default:
        return Database::asyncUpdatePodcast(index.data(IdRole).toString(), this->roleNames().value(role), value);
    }
}

bool PodcastListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return this->setData(this->index(row), value, this->roleNames().key(role));
}

bool PodcastListModel::loading() const {
    return m_loading;
}

void PodcastListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QString PodcastListModel::searchQuery() const {
    return m_searchQuery;
}

void PodcastListModel::setSearchQuery(const QString &query) {
    if (query != this->searchQuery()) {
        m_searchQuery = query;
        emit searchQueryChanged(query);
    }
}

Queries::QueryType PodcastListModel::queryType() const {
    return m_queryType;
}

void PodcastListModel::setQueryType(Queries::QueryType queryType) {
    if (queryType != this->queryType()) {
        m_queryType = queryType;
        emit queryTypeChanged(queryType);
    }
}

Station* PodcastListModel::get(const QModelIndex &index) const {
    return this->get(index.row());
}

Station* PodcastListModel::get(int row) const {
    return (row >= 0) && (row < m_list.size()) ? m_list.at(row) : 0;
}

void PodcastListModel::searchPodcasts(const QString &query) {
    this->setLoading(true);
    this->setSearchQuery(query);
    this->setQueryType(Queries::PodcastSearch);
    this->connect(Database::instance(), SIGNAL(gotPodcasts(QList<Station*>)), this, SLOT(addPodcasts(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetPodcastsByTitle", Qt::QueuedConnection, Q_ARG(QString, query));
}

void PodcastListModel::showPodcasts() {
    this->setLoading(true);
    this->setQueryType(Queries::Unknown);
    this->connect(Database::instance(), SIGNAL(gotPodcasts(QList<Station*>)), this, SLOT(addPodcasts(QList<Station*>)));
    QMetaObject::invokeMethod(Database::instance(), "asyncGetPodcasts", Qt::QueuedConnection);
}

void PodcastListModel::clear() {
    this->beginResetModel();
    qDeleteAll(m_list);
    m_list.clear();
    this->endResetModel();
}

void PodcastListModel::reload() {
    this->clear();

    switch (this->queryType()) {
    case Queries::PodcastSearch:
        this->searchPodcasts(this->searchQuery());
        return;
    default:
        this->showPodcasts();
        return;
    }
}

void PodcastListModel::addPodcasts(QList<Station *> podcasts) {
    this->disconnect(Database::instance(), SIGNAL(gotPodcasts(QList<Station*>)), this, SLOT(addPodcasts(QList<Station*>)));

    if (podcasts.isEmpty()) {
        this->setLoading(false);
        return;
    }

    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount() + podcasts.size() - 1);
    m_list << podcasts;
    this->endInsertRows();
    emit countChanged(this->rowCount());

    foreach (Station *podcast, podcasts) {
        this->connect(podcast, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastUpdated(QString,QVariantMap)));
        this->connect(podcast, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastUpdated(QString,QString,QVariant)));
    }

    this->setLoading(false);
}

void PodcastListModel::addPodcast(Station *podcast) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(podcast);
    this->endInsertRows();
    emit countChanged(this->rowCount());

    this->connect(podcast, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastUpdated(QString,QVariantMap)));
    this->connect(podcast, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastUpdated(QString,QString,QVariant)));
}

void PodcastListModel::insertPodcast(int row, Station *podcast) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginInsertRows(QModelIndex(), row, row);
        m_list.insert(row, podcast);
        this->endInsertRows();
        emit countChanged(this->rowCount());

        this->connect(podcast, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastUpdated(QString,QVariantMap)));
        this->connect(podcast, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastUpdated(QString,QString,QVariant)));
    }
}

void PodcastListModel::removePodcast(int row) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row)->deleteLater();
        this->endRemoveRows();
        emit countChanged(this->rowCount());
    }
}

void PodcastListModel::removePodcast(int role, const QVariant &value) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removePodcast(indexes.first().row());
    }
}

void PodcastListModel::onPodcastUpdated(const QString &id, const QVariantMap &properties) {
    Q_UNUSED(properties)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        emit dataChanged(indexes.first(), indexes.first());
    }
}

void PodcastListModel::onPodcastUpdated(const QString &id, const QString &property, const QVariant &value) {
    Q_UNUSED(property)
    Q_UNUSED(value)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        emit dataChanged(indexes.first(), indexes.first());
    }
}

void PodcastListModel::onPodcastDeleted(const QString &id) {
    this->removePodcast(IdRole, id);
}
