/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "podcastepisodelistmodel.h"
#include "podcasts.h"
#include "podcastepisodelist.h"
#ifndef QML_USER_INTERFACE
#include "../thumbnailcache/thumbnailcache.h"
#endif

PodcastEpisodeListModel::PodcastEpisodeListModel(QObject *parent) :
    QAbstractListModel(parent),
    #ifndef QML_USER_INTERFACE
    m_cache(new ThumbnailCache),
    #endif
    m_loading(false)
{
    m_roleNames[ServiceRole] = "service";
    m_roleNames[IdRole] = "id";
    m_roleNames[TitleRole] = "title";
    m_roleNames[DescriptionRole] = "description";
    m_roleNames[LogoRole] = "logo";
    m_roleNames[GenreRole] = "genre";
    m_roleNames[CountryRole] = "country";
    m_roleNames[LanguageRole] = "language";
    m_roleNames[SourceRole] = "source";
    m_roleNames[FavouriteRole] = "favourite";
    m_roleNames[LastPlayedRole] = "lastPlayed";
    m_roleNames[SectionRole] = "section";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

PodcastEpisodeListModel::~PodcastEpisodeListModel() {
    this->clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> PodcastEpisodeListModel::roleNames() const {
    return m_roleNames;
}
#endif

int PodcastEpisodeListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant PodcastEpisodeListModel::data(const QModelIndex &index, int role) const {
    if ((!this->rowCount()) || (!index.isValid())) {
        return QVariant();
    }

    switch (role) {
    case ServiceRole:
        return m_list.at(index.row())->service();
    case IdRole:
        return m_list.at(index.row())->id();
    case TitleRole:
        return m_list.at(index.row())->title();
    case DescriptionRole:
        return m_list.at(index.row())->description();
    case LogoRole:
#ifndef QML_USER_INTERFACE
        return m_cache->thumbnail(m_list.at(index.row())->logo(), QSize(64, 64));
#else
        return m_list.at(index.row())->logo();
#endif
    case GenreRole:
        return m_list.at(index.row())->genre();
    case CountryRole:
        return m_list.at(index.row())->country();
    case LanguageRole:
        return m_list.at(index.row())->language();
    case SourceRole:
        return m_list.at(index.row())->source();
    case FavouriteRole:
        return m_list.at(index.row())->favourite();
    case LastPlayedRole:
        return m_list.at(index.row())->lastPlayed();
    case SectionRole:
        return m_list.at(index.row())->title().left(1).toUpper();
    default:
        return QVariant();
    }
}

QVariant PodcastEpisodeListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool PodcastEpisodeListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if ((!this->rowCount()) || (!index.isValid())) {
        return false;
    }

    switch (role) {
    case IdRole:
    case LastPlayedRole:
    case SectionRole:
        break;
    default:
        if (Station *episode = this->get(index.row())) {
            return episode->setProperty(this->roleNames().value(role), value);
        }
    }

    return false;
}

bool PodcastEpisodeListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return this->setData(this->index(row), value, this->roleNames().key(role));
}

bool PodcastEpisodeListModel::loading() const {
    return m_loading;
}

void PodcastEpisodeListModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

Station* PodcastEpisodeListModel::get(const QModelIndex &index) const {
    return this->get(index.row());
}

Station* PodcastEpisodeListModel::get(int row) const {
    return (row >= 0) && (row < m_list.size()) ? m_list.at(row) : 0;
}

void PodcastEpisodeListModel::showPodcastEpisodes(const QString &url) {
    this->setLoading(true);

    PodcastEpisodeList *list = Podcasts::getPodcastEpisodes(url);
    list->setParent(this);
    this->connect(list, SIGNAL(finished(PodcastEpisodeList*)), this, SLOT(onPodcastEpisodeListFinished(PodcastEpisodeList*)));
    this->connect(list, SIGNAL(canceled(PodcastEpisodeList*)), this, SLOT(onPodcastEpisodeListCanceled(PodcastEpisodeList*)));
}

void PodcastEpisodeListModel::clear() {
    this->beginResetModel();
    qDeleteAll(m_list);
    m_list.clear();
    this->endResetModel();
}

void PodcastEpisodeListModel::reload() {
    this->clear();
    this->showPodcastEpisodes(QString());
}

void PodcastEpisodeListModel::onPodcastEpisodeListFinished(PodcastEpisodeList *list) {
    switch (list->error()) {
    case QNetworkReply::NoError:
        break;
    default:
        this->setLoading(false);
        emit error(list->errorString());
        list->deleteLater();
        return;
    }

    this->addPodcastEpisodes(list->results());
    list->deleteLater();
}

void PodcastEpisodeListModel::onPodcastEpisodeListCanceled(PodcastEpisodeList *list) {
    this->setLoading(false);
    list->deleteLater();
}

void PodcastEpisodeListModel::addPodcastEpisodes(QList<Station *> episodes) {
    if (episodes.isEmpty()) {
        this->setLoading(false);
        return;
    }

    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount() + episodes.size() - 1);
    m_list << episodes;
    this->endInsertRows();
    emit countChanged(this->rowCount());

    foreach (Station *episode, episodes) {
        this->connect(episode, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastEpisodeUpdated(QString,QVariantMap)));
        this->connect(episode, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastEpisodeUpdated(QString,QString,QVariant)));
    }

    this->setLoading(false);
}

void PodcastEpisodeListModel::addPodcastEpisode(Station *episode) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(episode);
    this->endInsertRows();
    emit countChanged(this->rowCount());

    this->connect(episode, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastEpisodeUpdated(QString,QVariantMap)));
    this->connect(episode, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastEpisodeUpdated(QString,QString,QVariant)));
}

void PodcastEpisodeListModel::insertPodcastEpisode(int row, Station *episode) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginInsertRows(QModelIndex(), row, row);
        m_list.insert(row, episode);
        this->endInsertRows();
        emit countChanged(this->rowCount());

        this->connect(episode, SIGNAL(stationUpdated(QString,QVariantMap)), this, SLOT(onPodcastEpisodeUpdated(QString,QVariantMap)));
        this->connect(episode, SIGNAL(stationUpdated(QString,QString,QVariant)), this, SLOT(onPodcastEpisodeUpdated(QString,QString,QVariant)));
    }
}

void PodcastEpisodeListModel::removePodcastEpisode(int row) {
    if ((row >= 0) && (row < this->rowCount())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row)->deleteLater();
        this->endRemoveRows();
        emit countChanged(this->rowCount());
    }
}

void PodcastEpisodeListModel::removePodcastEpisode(int role, const QVariant &value) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        this->removePodcastEpisode(indexes.first().row());
    }
}

void PodcastEpisodeListModel::onPodcastEpisodeUpdated(const QString &id, const QVariantMap &properties) {
    Q_UNUSED(properties)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        emit dataChanged(indexes.first(), indexes.first());
    }
}

void PodcastEpisodeListModel::onPodcastEpisodeUpdated(const QString &id, const QString &property, const QVariant &value) {
    Q_UNUSED(property)
    Q_UNUSED(value)

    QModelIndexList indexes = this->match(this->index(0), IdRole, id, 1, Qt::MatchExactly);

    if (!indexes.isEmpty()) {
        emit dataChanged(indexes.first(), indexes.first());
    }
}

void PodcastEpisodeListModel::onPodcastEpisodeDeleted(const QString &id) {
    this->removePodcastEpisode(IdRole, id);
}

#ifndef QML_USER_INTERFACE
void PodcastEpisodeListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif
