/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QtSql>
#include <QCoreApplication>
#include <QThread>
#include <QDir>
#include <qplatformdefs.h>
#include <QDebug>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif
#include "database.h"

Database* Database::self = 0;

Database::Database(QObject *parent) :
    QObject(parent)
{
    if (!self) {
        self = this;
    }

    this->moveToThread(new QThread);
    this->thread()->start();

    QDir dir;
#ifdef Q_OS_SYMBIAN
    dir.setPath(QCoreApplication::applicationDirPath() + "/stations/");
    dir.remove(QDir::toNativeSeparators(dir.path() + "/cuteradio.db"));
#elif (defined MEEGO_EDITION_HARMATTAN) || (defined Q_WS_MAEMO_5)
    dir.setPath("/home/user/cuteRadio/");
#elif QT_VERSION >= 0x050000
    dir.setPath(QStandardPaths::writeableLocation(QStandardPaths::HomeLocation) + "/cuteRadio/");
#else
    dir.setPath(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/cuteRadio/");
#endif
    if (!dir.mkpath(dir.path())) {
        qWarning() << "Cannot create path for the database";
        return;
    }

    QSqlDatabase database = QSqlDatabase::addDatabase("QSQLITE");
    database.setDatabaseName(QDir::toNativeSeparators(dir.path() + "/stations.db"));

    if (database.open()) {
        database.exec("CREATE TABLE IF NOT EXISTS stations (id TEXT UNIQUE, title TEXT, description TEXT, logo TEXT, genre TEXT, country TEXT, language TEXT, source TEXT, favourite INTEGER, lastPlayed TEXT)");
        database.exec("CREATE TABLE IF NOT EXISTS podcasts (id TEXT UNIQUE, title TEXT, description TEXT, logo TEXT, source TEXT, service INTEGER)");
    }
    else {
        qWarning() << "Cannot open database:" << database.lastError().text();
    }
}

Database::~Database() {
    this->thread()->quit();
    this->thread()->deleteLater();
}

Database* Database::instance() {
    return !self ? new Database : self;
}

// Stations
bool Database::addStation(const QString &title,
                          const QString &description,
                          const QUrl &logo,
                          const QString &genre,
                          const QString &country,
                          const QString &language,
                          const QUrl &source,
                          bool favourite) const {

    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("INSERT INTO stations VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    qsrand(QDateTime::currentMSecsSinceEpoch());
    int id = qrand();
    query.addBindValue(id);
    query.addBindValue(title);
    query.addBindValue(description);
    query.addBindValue(logo);
    query.addBindValue(genre);
    query.addBindValue(country);
    query.addBindValue(language);
    query.addBindValue(source);
    query.addBindValue(favourite ? 1 : 0);
    query.addBindValue(0);

    if (query.exec()) {
        emit stationAdded();
        emit alert(QString("%1 '%2' %3").arg(tr("New station")).arg(title).arg(tr("added")));

        return true;
    }

    emit error(tr("Cannot add new station"));

    return false;
}

bool Database::updateStation(const QString &id, const QVariantMap &properties, bool notify) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    QString queryString("UPDATE stations SET ");
    QMapIterator<QString, QVariant>iterator(properties);

    while (iterator.hasNext()) {
        iterator.next();
        queryString.append(iterator.hasNext() ? QString("%1 = '%2', ").arg(iterator.key()).arg(iterator.value().toString())
                                              : QString("%1 = '%2' ").arg(iterator.key()).arg(iterator.value().toString()));

    }

    queryString.append("WHERE id = " + id);

    if (query.exec(queryString)) {
        emit stationUpdated(id, properties);

        if (notify) {
            emit alert(tr("Station updated"));
        }

        return true;
    }

    if (notify) {
        emit error(tr("Cannot update station"));
    }

    return false;
}

bool Database::updateStation(const QString &id, const QString &property, const QVariant &value, bool notify) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare(QString("UPDATE stations SET %1 = ? WHERE id = ?").arg(property));
    query.addBindValue(value);
    query.addBindValue(id);

    if (query.exec()) {
        emit stationUpdated(id, property, value);

        if (notify) {
            emit alert(tr("Station updated"));
        }

        return true;
    }

    if (notify) {
        emit error(tr("Cannot update station"));
    }

    return false;
}

bool Database::deleteStation(const QString &id) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("DELETE FROM stations WHERE id = ?");
    query.addBindValue(id);

    if (query.exec()) {
        emit stationDeleted(id);
        emit alert(tr("Station deleted"));

        return true;
    }

    emit error(tr("Cannot delete station"));

    return false;
}

bool Database::asyncAddStation(const QString &title,
                               const QString &description,
                               const QUrl &logo,
                               const QString &genre,
                               const QString &country,
                               const QString &language,
                               const QUrl &source,
                               bool favourite) {

    return QMetaObject::invokeMethod(Database::instance(), "addStation", Qt::QueuedConnection,
                                     Q_ARG(QString, title),
                                     Q_ARG(QString, description),
                                     Q_ARG(QUrl, logo),
                                     Q_ARG(QString, genre),
                                     Q_ARG(QString, country),
                                     Q_ARG(QString, language),
                                     Q_ARG(QUrl, source),
                                     Q_ARG(bool, favourite));
}

bool Database::asyncUpdateStation(const QString &id, const QVariantMap &properties, bool notify) {
    return QMetaObject::invokeMethod(Database::instance(), "updateStation", Qt::QueuedConnection,
                                     Q_ARG(QString, id), Q_ARG(QVariantMap, properties), Q_ARG(bool, notify));
}

bool Database::asyncUpdateStation(const QString &id, const QString &property, const QVariant &value, bool notify) {
    return QMetaObject::invokeMethod(Database::instance(), "updateStation", Qt::QueuedConnection,
                                     Q_ARG(QString, id), Q_ARG(QString, property), Q_ARG(QVariant, value), Q_ARG(bool, notify));
}

bool Database::asyncDeleteStation(const QString &id) {
    return QMetaObject::invokeMethod(Database::instance(), "deleteStation", Qt::QueuedConnection, Q_ARG(QString, id));
}

QList<Station*> Database::getStations() const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations ORDER BY title ASC");

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getRecentlyPlayedStations() const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE lastPlayed > 0 ORDER BY lastPlayed DESC");

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getFavouriteStations() const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE favourite = 1 ORDER BY title ASC");

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsById(const QStringList &ids) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        QString queryString("SELECT * FROM stations WHERE id = ");

        for (int i = 0; i < ids.size(); i++) {
            queryString.append(i < (ids.size() - 1) ? ids.at(i) + " OR id = " : ids.at(i));
        }

        if (query.exec(queryString)) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByTitle(const QString &title) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec(QString("SELECT * FROM stations WHERE title LIKE '%%1%' OR description LIKE '%%1%' ORDER BY title ASC").arg(title))) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByGenre(const QString &genre) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE genre = ? ORDER BY title ASC");
        query.addBindValue(genre);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByCountry(const QString &country) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE country = ? ORDER BY title ASC");
        query.addBindValue(country);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByLanguage(const QString &language) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE language = ? ORDER BY title ASC");
        query.addBindValue(language);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(Stations::Radio,
                                                   Services::NoService,
                                                   query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   QUrl::fromLocalFile(query.value(3).toString()),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toString(),
                                                   query.value(7).toUrl(),
                                                   query.value(8).toBool(),
                                                   query.value(9).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

void Database::asyncGetStations() const {
    emit gotStations(this->getStations());
}

void Database::asyncGetRecentlyPlayedStations() const {
    emit gotStations(this->getRecentlyPlayedStations());
}

void Database::asyncGetFavouriteStations() const {
    emit gotStations(this->getFavouriteStations());
}

void Database::asyncGetStationsById(const QStringList &ids) const {
    emit gotStations(this->getStationsById(ids));
}

void Database::asyncGetStationsByTitle(const QString &title) const {
    emit gotStations(this->getStationsByTitle(title));
}

void Database::asyncGetStationsByGenre(const QString &genre) const {
    emit gotStations(this->getStationsByGenre(genre));
}

void Database::asyncGetStationsByCountry(const QString &country) const {
    emit gotStations(this->getStationsByCountry(country));
}

void Database::asyncGetStationsByLanguage(const QString &language) const {
    emit gotStations(this->getStationsByLanguage(language));
}

// Podcasts
bool Database::addPodcast(const QString &title,
                          const QString &description,
                          const QUrl &logo,
                          const QUrl &source,
                          int service) const {

    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("INSERT INTO podcasts VALUES (?, ?, ?, ?, ?, ?)");
    qsrand(QDateTime::currentMSecsSinceEpoch());
    int id = qrand();
    query.addBindValue(id);
    query.addBindValue(title);
    query.addBindValue(description);
    query.addBindValue(logo);
    query.addBindValue(source);
    query.addBindValue(service);

    if (query.exec()) {
        emit podcastAdded();
        emit alert(QString("%1 '%2' %3").arg(tr("New podcast")).arg(title).arg(tr("added")));

        return true;
    }

    emit error(tr("Cannot add new station"));

    return false;
}

bool Database::updatePodcast(const QString &id, const QVariantMap &properties, bool notify) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    QString queryString("UPDATE podcasts SET ");
    QMapIterator<QString, QVariant>iterator(properties);

    while (iterator.hasNext()) {
        iterator.next();
        queryString.append(iterator.hasNext() ? QString("%1 = '%2', ").arg(iterator.key()).arg(iterator.value().toString())
                                              : QString("%1 = '%2' ").arg(iterator.key()).arg(iterator.value().toString()));

    }

    queryString.append("WHERE id = " + id);

    if (query.exec(queryString)) {
        emit podcastUpdated(id, properties);

        if (notify) {
            emit alert(tr("Podcast updated"));
        }

        return true;
    }

    if (notify) {
        emit error(tr("Cannot update podcast"));
    }

    return false;
}

bool Database::updatePodcast(const QString &id, const QString &property, const QVariant &value, bool notify) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare(QString("UPDATE podcasts SET %1 = ? WHERE id = ?").arg(property));
    query.addBindValue(value);
    query.addBindValue(id);

    if (query.exec()) {
        emit podcastUpdated(id, property, value);

        if (notify) {
            emit alert(tr("Podcast updated"));
        }

        return true;
    }

    if (notify) {
        emit error(tr("Cannot update podcast"));
    }

    return false;
}

bool Database::deletePodcast(const QString &id) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("DELETE FROM podcasts WHERE id = ?");
    query.addBindValue(id);

    if (query.exec()) {
        emit podcastDeleted(id);
        emit alert(tr("Podcast deleted"));

        return true;
    }

    emit error(tr("Cannot delete podcast"));

    return false;
}

bool Database::asyncAddPodcast(const QString &title,
                               const QString &description,
                               const QUrl &logo,
                               const QUrl &source,
                               int service) {

    return QMetaObject::invokeMethod(Database::instance(), "addPodcast", Qt::QueuedConnection,
                                     Q_ARG(QString, title),
                                     Q_ARG(QString, description),
                                     Q_ARG(QUrl, logo),
                                     Q_ARG(QUrl, source),
                                     Q_ARG(int, service));
}

bool Database::asyncUpdatePodcast(const QString &id, const QVariantMap &properties, bool notify) {
    return QMetaObject::invokeMethod(Database::instance(), "updatePodcast", Qt::QueuedConnection,
                                     Q_ARG(QString, id), Q_ARG(QVariantMap, properties), Q_ARG(bool, notify));
}

bool Database::asyncUpdatePodcast(const QString &id, const QString &property, const QVariant &value, bool notify) {
    return QMetaObject::invokeMethod(Database::instance(), "updatePodcast", Qt::QueuedConnection,
                                     Q_ARG(QString, id), Q_ARG(QString, property), Q_ARG(QVariant, value), Q_ARG(bool, notify));
}

bool Database::asyncDeletePodcast(const QString &id) {
    return QMetaObject::invokeMethod(Database::instance(), "deletePodcast", Qt::QueuedConnection, Q_ARG(QString, id));
}

QList<Station*> Database::getPodcasts() const {
    QList<Station*> podcasts;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec("SELECT * FROM podcasts ORDER BY title ASC")) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *podcast = new Station;
                    podcast->setStationType(Stations::Podcast);
                    podcast->setId(query.value(0).toString());
                    podcast->setTitle(query.value(1).toString());
                    podcast->setDescription(query.value(2).toString());
                    podcast->setLogo(QUrl::fromLocalFile(query.value(3).toString()));
                    podcast->setSource(query.value(4).toUrl());
                    podcast->setService(static_cast<Services::RadioService>(query.value(5).toInt()));

                    podcasts.append(podcast);
                }
            }
        }
    }

    return podcasts;
}

QList<Station*> Database::getPodcastsById(const QStringList &ids) const {
    QList<Station*> podcasts;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        QString queryString("SELECT * FROM podcasts WHERE id = ");

        for (int i = 0; i < ids.size(); i++) {
            queryString.append(i < (ids.size() - 1) ? ids.at(i) + " OR id = " : ids.at(i));
        }

        if (query.exec(queryString)) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *podcast = new Station;
                    podcast->setStationType(Stations::Podcast);
                    podcast->setId(query.value(0).toString());
                    podcast->setTitle(query.value(1).toString());
                    podcast->setDescription(query.value(2).toString());
                    podcast->setLogo(QUrl::fromLocalFile(query.value(3).toString()));
                    podcast->setSource(query.value(4).toUrl());
                    podcast->setService(static_cast<Services::RadioService>(query.value(5).toInt()));

                    podcasts.append(podcast);
                }
            }
        }
    }

    return podcasts;
}

QList<Station*> Database::getPodcastsByTitle(const QString &title) const {
    QList<Station*> podcasts;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec(QString("SELECT * FROM podcasts WHERE title LIKE '%%1%' OR description LIKE '%%1%' ORDER BY title ASC").arg(title))) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *podcast = new Station;
                    podcast->setStationType(Stations::Podcast);
                    podcast->setId(query.value(0).toString());
                    podcast->setTitle(query.value(1).toString());
                    podcast->setDescription(query.value(2).toString());
                    podcast->setLogo(QUrl::fromLocalFile(query.value(3).toString()));
                    podcast->setSource(query.value(4).toUrl());
                    podcast->setService(static_cast<Services::RadioService>(query.value(5).toInt()));

                    podcasts.append(podcast);
                }
            }
        }
    }

    return podcasts;
}

void Database::asyncGetPodcasts() const {
    emit gotPodcasts(this->getPodcasts());
}

void Database::asyncGetPodcastsById(const QStringList &ids) const {
    emit gotPodcasts(this->getPodcastsById(ids));
}

void Database::asyncGetPodcastsByTitle(const QString &title) const {
    emit gotPodcasts(this->getPodcastsByTitle(title));
}
