/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.items 1.0

MySheet {
    id: root

    function loadStation(stationItem) {
        root.state = "loading";
        station.load(stationItem.id);
    }

    function addStation() {
        Database.asyncAddStation(titleField.text,
                            !descriptionField.text ? qsTr("No description") : descriptionField.text,
                            downloader.localPath,
                            !genreField.text ? qsTr("Unknown genre") : genreField.text,
                            !countryField.text ? qsTr("Unknown country") : countryField.text,
                            !languageField.text ? qsTr("Unknown language") : languageField.text,
                            sourceField.text,
                            favouriteCheckBox.checked);
        clear();
    }

    function clear() {
        if (station.result) {
            station.result.clear();
        }
    }

    TuneInStation {
        id: station

        onFinished: {
            if (station.result) {
                root.state = "finished";
            }
            else {
                root.state = "error";
                infoBanner.displayMessage(station.errorString);
            }
        }
    }

    LogoDownloader {
        id: downloader

        url: station.result ? station.result.logo : ""
        localPath: station.result ? Utils.logoPath + station.result.id + "." + station.result.logo.toString().split(".").pop() : ""
        onFinished: {
            progressDialog.close();
            root.addStation();
        }
    }

    hideHeaderWhenInputContextIsVisible: true
    acceptButtonText: (titleField.text) && (sourceField.text) && (sourceField.acceptableInput) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent

        KeyNavFlickable {
            id: flicker

            anchors {
                fill: parent
                margins: platformStyle.paddingLarge
            }
            contentHeight: inputContext.visible ? height : column.height + platformStyle.paddingLarge

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: platformStyle.paddingLarge

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Title")
                    visible: titleField.visible
                }

                MyTextField {
                    id: titleField

                    width: parent.width
                    text: station.result ? station.result.title : ""
                    visible: (!inputContext.visible) || (focus)
                    onAccepted: descriptionField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Description")
                    visible: descriptionField.visible
                }

                TextArea {
                    id: descriptionField

                    width: parent.width
                    text: station.result ? station.result.description : ""
                    visible: (!inputContext.visible) || (focus)
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Genre")
                    visible: genreField.visible
                }

                MyTextField {
                    id: genreField

                    width: parent.width
                    text: station.result ? station.result.genre : ""
                    visible: (!inputContext.visible) || (focus)
                    onAccepted: countryField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Country")
                    visible: countryField.visible
                }

                MyTextField {
                    id: countryField

                    width: parent.width
                    text: station.result ? station.result.country : ""
                    visible: (!inputContext.visible) || (focus)
                    onAccepted: languageField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Language")
                    visible: languageField.visible
                }

                MyTextField {
                    id: languageField

                    width: parent.width
                    text: station.result ? station.result.language : ""
                    visible: (!inputContext.visible) || (focus)
                    onAccepted: sourceField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                    visible: sourceField.visible
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    text: station.result ? station.result.source : ""
                    visible: (!inputContext.visible) || (focus)
                    validator: RegExpValidator {
                        regExp: /^http(s|):\/\/.+/
                    }
                    onAccepted: closeSoftwareInputPanel()
                }

                MyCheckBox {
                    id: favouriteCheckBox

                    text: qsTr("Favourite?")
                    checked: station.result ? station.result.favourite : false
                    visible: !inputContext.visible
                }
            }
        }

        MyScrollBar {
            flickableItem: flicker
        }

        Label {
            id: errorLabel

            anchors {
                fill: parent
                margins: platformStyle.paddingLarge
            }
            horizontalAlignment: Text.AlignHCenter
            verticalAlignment: Text.AlignVCenter
            wrapMode: Text.WordWrap
            color: platformStyle.colorNormalMid
            font.bold: true
            font.pixelSize: 32
            text: qsTr("No information found")
            visible: false
        }

        BusyIndicator {
            id: busyIndicator

            anchors.centerIn: parent
            width: 64
            height: 64
            visible: false
            running: visible
        }
    }

    states: [
        State {
            name: "loading"
            PropertyChanges { target: flicker; visible: false }
            PropertyChanges { target: busyIndicator; visible: true }
        },

        State {
            name: "finished"
            PropertyChanges { target: flicker; visible: true }
            PropertyChanges { target: busyIndicator; visible: false }
        },

        State {
            name: "error"
            PropertyChanges { target: flicker; visible: false }
            PropertyChanges { target: busyIndicator; visible: false }
            PropertyChanges { target: errorLabel; visible: true }
        }
    ]

    Connections {
        target: progressDialog
        onRejected: {
            downloader.cancel();
            clear();
        }
    }

    onAccepted: {
        progressDialog.showBusyMessage(qsTr("Adding station"), 1);
        downloader.start();
    }
    onRejected: clear()
}
