/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.items 1.0

MySheet {
    id: root

    function addPodcast() {
        Database.asyncAddPodcast(podcast.result.title,
                            !podcast.result.description ? qsTr("No description") : podcast.result.description,
                            !podcast.result.logo ? "" : downloader.localPath,
                            podcast.result.source);
        clear();
    }

    function clear() {
        sourceField.text = "";
    }

    Podcast {
        id: podcast

        onFinished: {
            if (result) {
                if (result.logo) {
                    downloader.url = result.logo;
                    downloader.localPath = Utils.logoPath + (result.id ? result.id :result.title) + "." + result.logo.toString().split(".").pop();
                    downloader.start();
                }
                else {
                    progressDialog.close();
                    root.addPodcast();
                }
            }
            else {
                progressDialog.close();
                infoBanner.displayMessage(podcast.errorString);
            }
        }
    }

    LogoDownloader {
        id: downloader

        onFinished: {
            progressDialog.close();
            root.addPodcast();
        }
    }

    hideHeaderWhenInputContextIsVisible: true
    acceptButtonText: (sourceField.text) && (sourceField.acceptableInput) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent
        clip: true

        KeyNavFlickable {
            id: flicker

            anchors {
                fill: parent
                margins: platformStyle.paddingLarge
            }
            contentHeight: column.height + platformStyle.paddingLarge

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: platformStyle.paddingLarge

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    validator: RegExpValidator {
                        regExp: /^http(s|):\/\/.+/
                    }
                    onAccepted: root.accept()
                }
            }
        }

        MyScrollBar {
            flickableItem: flicker
        }
    }

    Connections {
        target: progressDialog
        onRejected: {
            podcast.cancel();
            downloader.cancel();
            clear();
        }
    }

    onAccepted: {
        progressDialog.showBusyMessage(qsTr("Adding podcast"), 1);
        podcast.load(sourceField.text);
    }
    onRejected: clear()
}
