/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.meego 1.0
import com.cuteradio.items 1.0
import "file:///usr/lib/qt4/imports/com/nokia/meego/UIConstants.js" as UI

MySheet {
    id: root

    function loadStation(stationItem) {
        station.loadStation(stationItem);
    }

    function addStation() {
        Database.asyncAddStation(titleField.text,
                            !descriptionField.text ? qsTr("No description") : descriptionField.text,
                            !logoField.text ? "" : downloader.localPath,
                            !genreField.text ? qsTr("Unknown genre") : genreField.text,
                            !countryField.text ? qsTr("Unknown country") : countryField.text,
                            !languageField.text ? qsTr("Unknown language") : languageField.text,
                            sourceField.text,
                            favouriteCheckBox.checked);
        clear();
    }

    function updateStation() {
        var properties = {};

        if (titleField.text != station.title) {
            properties["title"] = titleField.text;
        }
        if (descriptionField.text != station.description) {
            properties["description"] = !descriptionField.text ? qsTr("No description") : descriptionField.text;
        }
        if (logoField.text != station.logo) {
            properties["logo"] = !logoField.text ? "" : downloader.localPath;
        }
        if (genreField.text != station.genre) {
            properties["genre"] = !genreField.text ? qsTr("Unknown genre") : genreField.text;
        }
        if (countryField.text != station.country) {
            properties["country"] = !countryField.text ? qsTr("Unknown country") : countryField.text;
        }
        if (languageField.text != station.language) {
            properties["language"] = !languageField.text ? qsTr("Unknown language") : languageField.text;
        }
        if (sourceField.text != station.source) {
            properties["source"] = sourceField.text;
        }
        if (favouriteCheckBox.checked != station.favourite) {
            properties["favourite"] = favouriteCheckBox.checked ? 1 : 0;
        }

        Database.asyncUpdateStation(station.id, properties);

        clear();
    }

    function clear() {
        if (station.id) {
            station.clear();
        }
        else {
            titleField.text = "";
            descriptionField.text = "";
            logoField.text = "";
            genreField.text = "";
            countryField.text = "";
            languageField.text = "";
            sourceField.text = "";
            favouriteCheckBox.checked = false;
        }
    }

    Station {
        id: station
    }

    LogoDownloader {
        id: downloader

        url: logoField.text
        localPath: Utils.logoPath + (station.id ? station.id : station.title) + "." + logoField.text.split(".").pop()
        onFinished: {
            progressDialog.close();

            if (station.id) {
                root.updateStation();
            }
            else {
                root.addStation();
            }
        }
    }

    acceptButtonText: (titleField.text) && (sourceField.text) && (sourceField.acceptableInput) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent
        clip: true

        Flickable {
            id: flicker

            anchors {
                fill: parent
                margins: UI.PADDING_DOUBLE
            }
            contentHeight: column.height + UI.PADDING_DOUBLE

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: UI.PADDING_DOUBLE

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Title")
                }

                MyTextField {
                    id: titleField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.title
                    onAccepted: descriptionField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Description")
                }

                MyTextArea {
                    id: descriptionField

                    width: parent.width
                    text: station.description
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Logo")
                }

                MyTextField {
                    id: logoField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.logo
                    onAccepted: genreField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Genre")
                }

                MyTextField {
                    id: genreField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.genre
                    onAccepted: countryField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Country")
                }

                MyTextField {
                    id: countryField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.country
                    onAccepted: languageField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Language")
                }

                MyTextField {
                    id: languageField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: station.language
                    onAccepted: sourceField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    text: station.source
                    validator: RegExpValidator {
                        regExp: /^http(s|):\/\/.+/
                    }
                    onAccepted: platformCloseSoftwareInputPanel()
                }

                MyCheckBox {
                    id: favouriteCheckBox

                    text: qsTr("Favourite?")
                    checked: station.favourite
                }
            }
        }

        ScrollDecorator {
            flickableItem: flicker
        }
    }

    Connections {
        target: progressDialog
        onRejected: {
            downloader.cancel();
            clear();
        }
    }

    onAccepted: {
        if (downloader.localPath.substring(0, 4) == "http") {
            progressDialog.showBusyMessage(station.id ? qsTr("Updating station") : qsTr("Adding station"), 1);
            downloader.start();
        }
        else if (station.id) {
            updateStation();
        }
        else {
            addStation();
        }
    }
    onRejected: clear()
}
