/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QDomDocument>
#include <QDomElement>
#include "tuneinnamelist.h"

TuneInNameList::TuneInNameList(QNetworkReply *reply, QObject *parent) :
    QObject(parent),
    m_reply(reply)
{
    this->connect(reply, SIGNAL(finished()), this, SLOT(onReplyFinished()));
}

TuneInNameList::~TuneInNameList() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

int TuneInNameList::count() const {
    return m_results.size();
}

QList< QPair<QString, QString> > TuneInNameList::results() const {
    return m_results;
}

QNetworkReply::NetworkError TuneInNameList::error() const {
    return m_reply ? m_reply->error() : QNetworkReply::NoError;
}

QString TuneInNameList::errorString() const {
    return m_reply ? m_reply->errorString() : QString();
}

void TuneInNameList::cancel() {
    if (m_reply) {
        m_reply->abort();
    }
    else {
        emit canceled(this);
    }
}

void TuneInNameList::onReplyFinished() {
    if (!m_reply) {
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        emit canceled(this);
        return;
    default:
        emit finished(this);
        return;
    }

    QDomDocument doc;
    doc.setContent(m_reply->readAll());

    QDomNodeList results = doc.documentElement().firstChildElement("body").elementsByTagName("outline");

    for (int i = 0; i < results.size(); i++) {
        QDomElement result = results.at(i).toElement();

        if (result.attribute("type") == "link") {
            m_results.append(QPair<QString, QString>(result.attribute("text"), result.attribute("URL")));
        }
    }

    emit finished(this);
}
