/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QVBoxLayout>
#include <QMaemo5Style>
#include "tuneinpodcastlistwindow.h"
#include "tuneinpodcastepisodelistwindow.h"
#include "tuneinpodcastlistdelegate.h"
#include "listview.h"
#include "filterbox.h"
#include "nowplayingbutton.h"
#include "tuneinaddpodcastdialog.h"
#include "launcher.h"
#include "../base/utils.h"
#include "../base/settings.h"
#include "../base/streamextractor.h"
#include "../tunein/tuneinpodcastfilterlistmodel.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

TuneInPodcastListWindow::TuneInPodcastListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new TuneInPodcastFilterListModel(this)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_addAction(m_contextMenu->addAction(tr("Add to database"), this, SLOT(showAddPodcastDialog()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No podcasts found")), this))
{
    this->setWindowTitle(tr("Podcasts"));
    this->setCentralWidget(m_noResultsLabel);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new TuneInPodcastListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_filterBox->hide();

    m_vbox = new QVBoxLayout(m_noResultsLabel);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), this, SLOT(onScrollingStopped()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(cleared()), NowPlayingButton::instance(), SLOT(poke()));
}

void TuneInPodcastListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void TuneInPodcastListWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (m_model->rowCount() > 0) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        m_filterBox->show();
        m_filterBox->setFocus(Qt::OtherFocusReason);
        m_filterBox->setFilterText(event->text());
    }
}

void TuneInPodcastListWindow::searchPodcasts(const QString &query) {
    this->setWindowTitle(tr("Search ('%1')").arg(query));
    m_model->searchPodcasts(query);
}

void TuneInPodcastListWindow::showPodcastsByGenre(const QString &genre, const QString &id) {
    this->setWindowTitle(genre);
    m_model->showPodcastsByGenre(id);
}

void TuneInPodcastListWindow::showPodcasts(const QString &title, const QString &url) {
    this->setWindowTitle(title);
    m_model->showPodcasts(url);
}

void TuneInPodcastListWindow::clear() {
    m_model->clear();
}

void TuneInPodcastListWindow::reload() {
    m_model->reload();
}

void TuneInPodcastListWindow::onScrollingStopped() {
    if (m_filterBox->isHidden()) {
        NowPlayingButton::instance()->poke();
    }
}

void TuneInPodcastListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_view->setHidden((!loading) && (m_model->rowCount() == 0));
}

void TuneInPodcastListWindow::onCountChanged(int count) {
    m_view->setHidden((!m_model->loading()) && (count == 0));
}

void TuneInPodcastListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addAction);
}

void TuneInPodcastListWindow::showAddPodcastDialog() {
    if (m_view->currentIndex().isValid()) {
        TuneInAddPodcastDialog *dialog = new TuneInAddPodcastDialog(m_model->get(m_view->currentIndex()), this);
        dialog->open();
    }
}

void TuneInPodcastListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        TuneInPodcastEpisodeListWindow *window = new TuneInPodcastEpisodeListWindow(this);
        window->showPodcastEpisodes(index.data(TuneInPodcastListModel::TitleRole).toString(), index.data(TuneInPodcastListModel::SourceRole).toString());
        window->show();
        return;
    }
}
