/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QMessageBox>
#include <QVBoxLayout>
#include <QMaemo5Style>
#include "stationlistwindow.h"
#include "stationinfowindow.h"
#include "stationlistdelegate.h"
#include "listview.h"
#include "filterbox.h"
#include "nowplayingbutton.h"
#include "addstationdialog.h"
#include "launcher.h"
#include "../base/stationfilterlistmodel.h"
#include "../base/database.h"
#include "../base/settings.h"
#include "../base/utils.h"
#include "../base/streamextractor.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

StationListWindow::StationListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_view(new ListView(this)),
    m_model(new StationFilterListModel(this)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_infoAction(m_contextMenu->addAction(tr("View info"), this, SLOT(showStationInfo()))),
    m_favouriteAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(toggleFavourite()))),
    m_editAction(m_contextMenu->addAction(tr("Edit"), this, SLOT(showEditDialog()))),
    m_deleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deleteStation()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No stations found")), this))
{
    this->setWindowTitle(tr("Stations"));
    this->setCentralWidget(m_noResultsLabel);

    m_view->setModel(m_model);
    m_view->setItemDelegate(new StationListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_filterBox->hide();

    m_vbox = new QVBoxLayout(m_noResultsLabel);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_view);
    m_vbox->addWidget(m_filterBox);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), this, SLOT(onScrollingStopped()));
    this->connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));
    this->connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    this->connect(m_filterBox, SIGNAL(cleared()), NowPlayingButton::instance(), SLOT(poke()));
}

void StationListWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void StationListWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (m_model->rowCount() > 0) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        m_filterBox->show();
        m_filterBox->setFocus(Qt::OtherFocusReason);
        m_filterBox->setFilterText(event->text());
    }
}

void StationListWindow::searchStations(const QString &query) {
    this->setWindowTitle(tr("Search ('%1')").arg(query));
    m_model->searchStations(query);
}

void StationListWindow::showRecentlyPlayedStations() {
    this->setWindowTitle(tr("Recently played stations"));
    m_model->showRecentlyPlayedStations();
}

void StationListWindow::showFavouriteStations() {
    this->setWindowTitle(tr("Favourite stations"));
    m_model->showFavouriteStations();
}

void StationListWindow::showStationsByGenre(const QString &genre) {
    this->setWindowTitle(genre);
    m_model->showStationsByGenre(genre);
}

void StationListWindow::showStationsByCountry(const QString &country) {
    this->setWindowTitle(country);
    m_model->showStationsByCountry(country);
}

void StationListWindow::showStationsByLanguage(const QString &language) {
    this->setWindowTitle(language);
    m_model->showStationsByLanguage(language);
}

void StationListWindow::clear() {
    m_model->clear();
}

void StationListWindow::reload() {
    m_model->reload();
}

void StationListWindow::onScrollingStopped() {
    if (m_filterBox->isHidden()) {
        NowPlayingButton::instance()->poke();
    }
}

void StationListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_view->setHidden((!loading) && (m_model->rowCount() == 0));
}

void StationListWindow::onCountChanged(int count) {
    m_view->setHidden((!m_model->loading()) && (count == 0));
}

void StationListWindow::showContextMenu(const QPoint &pos) {
    m_favouriteAction->setText(m_view->currentIndex().data(StationListModel::FavouriteRole).toBool() ? tr("Delete from favourites") : tr("Add to favourites"));
    m_contextMenu->popup(pos, m_infoAction);
}

void StationListWindow::showStationInfo() {
    if (m_view->currentIndex().isValid()) {
        StationInfoWindow *window = new StationInfoWindow(m_model->get(m_view->currentIndex()), this);
        window->show();
    }
}

void StationListWindow::toggleFavourite() {
    if (m_view->currentIndex().isValid()) {
        Database::asyncUpdateStation(m_view->currentIndex().data(StationListModel::IdRole).toString(), "favourite",
                                     m_view->currentIndex().data(StationListModel::FavouriteRole).toBool() ? 0 : 1, false);
    }
}

void StationListWindow::showEditDialog() {
    if (m_view->currentIndex().isValid()) {
        AddStationDialog *dialog = new AddStationDialog(m_model->get(m_view->currentIndex()), this);
        dialog->open();
    }
}

void StationListWindow::deleteStation() {
    if ((m_view->currentIndex().isValid()) && (QMessageBox::question(this, tr("Delete station?"), tr("Do you wish to delete the station '%1' from the database?")
                                                                     .arg(m_view->currentIndex().data(StationListModel::TitleRole).toString()), QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)) {

        Database::asyncDeleteStation(m_view->currentIndex().data(StationListModel::IdRole).toString());
    }
}

void StationListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        if (Settings::instance()->mediaPlayer() == "cuteradio") {
            MediaPlayer::instance()->play(m_model->get(index));
        }
        else {
            QUrl url = index.data(StationListModel::SourceRole).toUrl();

            if (Utils::urlIsPlaylist(url)) {
                StreamExtractor::instance()->getStreamUrl(url);
            }
            else {
                Launcher::launch(url);
            }
        }
    }
}
