/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QLineEdit>
#include <QGridLayout>
#include <QRegExpValidator>
#include <QMenuBar>
#include <QProgressDialog>
#include <QMaemo5InformationBox>
#include "mainwindow.h"
#include "listview.h"
#include "homescreenlistdelegate.h"
#include "browselistwindow.h"
#include "stationlistwindow.h"
#include "podcastlistwindow.h"
#include "tuneinwindow.h"
#include "nowplayingbutton.h"
#include "valueselector.h"
#include "addstationdialog.h"
#include "addpodcastdialog.h"
#include "playstationdialog.h"
#include "settingsdialog.h"
#include "aboutdialog.h"
#include "../base/homescreenmodel.h"
#include "../base/searchlistmodel.h"
#include "../base/selectionmodels.h"
#include "../base/database.h"
#include "../base/streamextractor.h"
#include "../base/settings.h"
#ifdef MOBILITY_PLAYER
#include "../mediaplayer/mobility/mediaplayer.h"
#else
#include "../mediaplayer/phonon/mediaplayer.h"
#endif

MainWindow::MainWindow(QWidget *parent) :
    StackedWindow(parent),
    m_model(new HomescreenModel(this)),
    m_searchModel(new SearchListModel(this)),
    m_searchEdit(new QLineEdit(this)),
    m_searchTypeSelector(new ValueSelector(tr("Search for"), this)),
    m_view(new ListView(this)),
    m_nowPlayingButton(new NowPlayingButton(this)),
    m_progressDialog(0)
{
    this->setWindowTitle(tr("cuteRadio"));
    this->setCentralWidget(new QWidget(this));
    this->menuBar()->addAction(tr("Play from URL"), this, SLOT(showPlayStationDialog()));
    this->menuBar()->addAction(tr("Add station"), this, SLOT(showAddStationDialog()));
    this->menuBar()->addAction(tr("Add podcast"), this, SLOT(showAddPodcastDialog()));
    this->menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()));
    this->menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()));
    this->setAttribute(Qt::WA_DeleteOnClose, false);

    m_searchEdit->setPlaceholderText(tr("Search"));
    m_searchEdit->setValidator(new QRegExpValidator(QRegExp("^\\S.+"), m_searchEdit));
    m_searchEdit->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    m_searchTypeSelector->setModel(new StationTypeModel(m_searchTypeSelector));
    m_searchTypeSelector->setValue(Settings::instance()->searchType());

    m_view->setModel(m_model);
    m_view->setItemDelegate(new HomescreenListDelegate(m_view));
    m_view->setMonitorScrolling(true);

    m_grid = new QGridLayout(this->centralWidget());
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_searchEdit, 0, 0);
    m_grid->addWidget(m_searchTypeSelector, 0, 1);
    m_grid->addWidget(m_view, 1, 0, 1, 2);

    this->connect(m_searchEdit, SIGNAL(returnPressed()), this, SLOT(search()));
    this->connect(m_searchTypeSelector, SIGNAL(valueChanged(QVariant)), this, SLOT(onSearchTypeChanged(QVariant)));
    this->connect(m_view, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(m_view, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_view, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));
    this->connect(Database::instance(), SIGNAL(alert(QString)), this, SLOT(showInformationBanner(QString)));
    this->connect(Database::instance(), SIGNAL(error(QString)), this, SLOT(showInformationBanner(QString)));
    this->connect(MediaPlayer::instance(), SIGNAL(error(QString)), this, SLOT(showInformationBanner(QString)));
    this->connect(StreamExtractor::instance(), SIGNAL(error(QString)), this, SLOT(showInformationBanner(QString)));
    this->connect(StreamExtractor::instance(), SIGNAL(busy(QString,int)), this, SLOT(showProgressDialog(QString,int)));
    this->connect(StreamExtractor::instance(), SIGNAL(busyProgressChanged(int)), this, SLOT(updateProgressDialog(int)));
    this->connect(StreamExtractor::instance(), SIGNAL(currentOperationCanceled()), this, SLOT(cancelProgressDialog()));
}

void MainWindow::onItemClicked(const QModelIndex &index) {
    switch (index.row()) {
    case 0:
        this->showGenres();
        return;
    case 1:
        this->showCountries();
        return;
    case 2:
        this->showLanguages();
        return;
    case 3:
        this->showRecentlyPlayedStations();
        return;
    case 4:
        this->showFavouriteStations();
        return;
    case 5:
        this->showPodcasts();
        return;
    case 6:
        this->showTuneIn();
        return;
    default:
        return;
    }
}

void MainWindow::onSearchTypeChanged(const QVariant &type) {
    Settings::instance()->setSearchType(static_cast<Stations::StationType>(type.toInt()));
}

void MainWindow::search() {
    switch (m_searchTypeSelector->currentValue().toInt()) {
    case Stations::Podcast:
        this->searchPodcasts();
        return;
    default:
        this->searchStations();
        return;
    }
}

void MainWindow::searchStations() {
    StationListWindow *window = new StationListWindow(this);
    window->searchStations(m_searchEdit->text());
    window->show();
    m_searchModel->addSearch(m_searchEdit->text());
    m_searchEdit->clear();
}

void MainWindow::searchPodcasts() {
    PodcastListWindow *window = new PodcastListWindow(this);
    window->searchPodcasts(m_searchEdit->text());
    window->show();
    m_searchModel->addSearch(m_searchEdit->text());
    m_searchEdit->clear();
}

void MainWindow::showGenres() {
    BrowseListWindow *window = new BrowseListWindow(this);
    window->showGenres();
    window->show();
}

void MainWindow::showCountries() {
    BrowseListWindow *window = new BrowseListWindow(this);
    window->showCountries();
    window->show();
}

void MainWindow::showLanguages() {
    BrowseListWindow *window = new BrowseListWindow(this);
    window->showLanguages();
    window->show();
}

void MainWindow::showRecentlyPlayedStations() {
    StationListWindow *window = new StationListWindow(this);
    window->showRecentlyPlayedStations();
    window->show();
}

void MainWindow::showFavouriteStations() {
    StationListWindow *window = new StationListWindow(this);
    window->showFavouriteStations();
    window->show();
}

void MainWindow::showPodcasts() {
    PodcastListWindow *window = new PodcastListWindow(this);
    window->showPodcasts();
    window->show();
}

void MainWindow::showTuneIn() {
    TuneInWindow *window = new TuneInWindow(this);
    window->show();
}

void MainWindow::showAddStationDialog() {
    AddStationDialog *dialog = new AddStationDialog(this);
    dialog->open();
}

void MainWindow::showAddPodcastDialog() {
    AddPodcastDialog *dialog = new AddPodcastDialog(this);
    dialog->open();
}

void MainWindow::showPlayStationDialog() {
    PlayStationDialog *dialog = new PlayStationDialog(this);
    dialog->open();
}

void MainWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->open();
}

void MainWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}

void MainWindow::showInformationBanner(const QString &message) {
    QMaemo5InformationBox::information(0, message);
}

void MainWindow::showProgressDialog(const QString &message, int numberOfOperations) {
    if (numberOfOperations == 0) {
        return;
    }

    if (!m_progressDialog) {
        m_progressDialog = new QProgressDialog;
        m_progressDialog->setWindowTitle(tr("Please wait"));
        m_progressDialog->setCancelButtonText(tr("Cancel"));
        m_progressDialog->setMinimumDuration(0);
        m_progressDialog->setMinimum(0);
    }

    switch (Settings::instance()->screenOrientation()) {
    case ScreenOrientation::LockLandscape:
        m_progressDialog->setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        break;
    case ScreenOrientation::LockPortrait:
        m_progressDialog->setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        break;
    default:
        m_progressDialog->setAttribute(Qt::WA_Maemo5AutoOrientation, true);
    }

    m_progressDialog->setLabelText(message);
    m_progressDialog->setMaximum(numberOfOperations);
    m_progressDialog->setValue(0);
    m_progressDialog->show();

    if (this->sender()) {
        this->connect(m_progressDialog, SIGNAL(canceled()), this->sender(), SLOT(cancelCurrentOperation()));
    }
}

void MainWindow::updateProgressDialog(int progress) {
    if ((m_progressDialog) && (!m_progressDialog->wasCanceled())) {
        m_progressDialog->setValue(progress);
    }
}

void MainWindow::cancelProgressDialog() {
    if (m_progressDialog) {
        m_progressDialog->reset();

        if (this->sender()) {
            this->disconnect(m_progressDialog, SIGNAL(canceled()), this->sender(), SLOT(cancelCurrentOperation()));
        }
    }
}
