/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QNetworkRequest>
#include <QNetworkReply>
#include <QRegExp>
#include <QDomDocument>
#include <QDomElement>
#include "streamextractor.h"
#include "utils.h"
#include "networkaccessmanager.h"

StreamExtractor* StreamExtractor::self = 0;

static const int MAX_RETRIES = 8;

StreamExtractor::StreamExtractor(QObject *parent) :
    QObject(parent),
    m_retries(0),
    m_busy(false),
    m_canceled(false)
{
    if (!self) {
        self = this;
    }
}

StreamExtractor::~StreamExtractor() {}

StreamExtractor* StreamExtractor::instance() {
    return !self ? new StreamExtractor : self;
}

static QString getUrlFromASXFile(const QString &response) {
    QDomDocument doc;
    doc.setContent(response.toLower());
    QDomNode node = doc.documentElement().namedItem("entry");

    return node.firstChildElement("ref").attribute("href");
}

static QString getUrlFromPLSFile(const QString &response) {
    return response.section(QRegExp("File\\d=", Qt::CaseInsensitive), 1, 1).section(QRegExp("\\s"), 0, 0);
}

static QString getUrlFromM3UFile(const QString &response) {
    QRegExp re("http(s|)://\\S+", Qt::CaseInsensitive);

    return re.indexIn(response) >= 0 ? re.cap() : QString();
}

static QString getUrlFromSMILFile(const QString &response) {
    QDomDocument doc;
    doc.setContent(response.toLower());
    QDomNode node = doc.documentElement().namedItem("body");

    return node.firstChildElement("audio").attribute("src");
}

static QString getUrlFromUnknownFile(const QString &response) {
    QRegExp re("http(s|)://[^\\s\"'<>]+", Qt::CaseInsensitive);

    return re.indexIn(response) >= 0 ? re.cap() : QString();
}

static QString getUrlFromPlaylistFile(const QString &response, const QString &format) {
    if (format == "asx") {
        return getUrlFromASXFile(response);
    }
    else if (format == "pls") {
        return getUrlFromPLSFile(response);
    }
    else if (format == "m3u") {
        return getUrlFromM3UFile(response);
    }
    else if (format == "smil") {
        return getUrlFromSMILFile(response);
    }
    else {
        return getUrlFromUnknownFile(response);
    }
}

int StreamExtractor::retries() const {
    return m_retries;
}

void StreamExtractor::setRetries(int retries) {
    m_retries = retries;
}

bool StreamExtractor::busy() const {
    return m_busy;
}

void StreamExtractor::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;
        emit busyChanged(isBusy);
    }

    if (isBusy) {
        this->setCanceled(false);
        emit busy(message, numberOfOperations);
    }
    else if (!this->canceled()) {
        emit busyProgressChanged(numberOfOperations);
    }
}

bool StreamExtractor::canceled() const {
    return m_canceled;
}

void StreamExtractor::setCanceled(bool canceled) {
    m_canceled = canceled;
}

void StreamExtractor::cancelCurrentOperation() {
    this->setCanceled(true);
    this->setBusy(false);
    emit currentOperationCanceled();
}

void StreamExtractor::getStreamUrl(const QUrl &url) {
    this->setBusy(true, tr("Retrieving stream URL"));
    this->setRetries(0);
    QNetworkRequest request(url);
    QNetworkReply *reply = NetworkAccessManager::instance()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseResponse()));
    this->connect(this, SIGNAL(currentOperationCanceled()), reply, SLOT(deleteLater()));
}

void StreamExtractor::retry(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = NetworkAccessManager::instance()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseResponse()));
    this->connect(this, SIGNAL(currentOperationCanceled()), reply, SLOT(deleteLater()));
}

void StreamExtractor::parseResponse() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    switch (reply->error()) {
    case QNetworkReply::NoError:
        break;
    case QNetworkReply::OperationCanceledError:
        reply->deleteLater();
        return;
    default:
        this->setBusy(false);
        emit error(reply->errorString());
        reply->deleteLater();
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        if (this->retries() < MAX_RETRIES) {
            this->setRetries(this->retries() + 1);
            this->retry(redirect);
        }
        else {
            this->setBusy(false);
            emit error(tr("Cannot find stream url"));
        }
    }
    else {
        QString response(reply->readAll());
        QString format = reply->url().toString().section('.', -1).toLower();
        QUrl url(getUrlFromPlaylistFile(response, format));

        if (url.isEmpty()) {
            this->setBusy(false);
            emit error(tr("Cannot find stream url"));
        }
        else {
            if (Utils::urlIsPlaylist(url)) {
                if (this->retries() < MAX_RETRIES) {
                    this->setRetries(this->retries() + 1);
                    this->retry(url);
                }
                else {
                    this->setBusy(false);
                    emit error(tr("Cannot find stream url"));
                }
            }
            else {
                this->setBusy(false);
                emit gotStreamUrl(url);
            }
        }
    }

    reply->deleteLater();
}
