/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.symbian 1.1
import com.cuteradio.items 1.0

MyPage {
    id: root

    function loadStation(stationItem) {
        station.loadStation(stationItem);
    }

    title: !station.title ? qsTr("Station Info") : station.title
    tools: ToolBarLayout {

        BackToolButton {}

        MyToolButton {
            iconSource: station.favourite ? "images/favourite-" + Settings.activeColorString + ".png"
                                          : "images/favourite-white.png"
            toolTipText: qsTr("Toggle favourite")
            onClicked: Database.asyncUpdateStation(station.id, "favourite", station.favourite ? 0 : 1, false)
        }

        MyToolButton {
            iconSource: "toolbar-mediacontrol-play"
            toolTipText: qsTr("Play")
            enabled: (!MediaPlayer.currentStation) || (MediaPlayer.currentStation.id != station.id)
            onClicked: Settings.mediaPlayer == "cuteradio" ? MediaPlayer.play(station)
                                                           : Utils.urlIsPlaylist(station.source)
                                                             ? StreamExtractor.getStreamUrl(station.source)
                                                             : Launcher.launch(station.source)
        }

        MyToolButton {
            iconSource: "images/edit.png"
            toolTipText: qsTr("Edit")
            onClicked: {
                loader.sourceComponent = editStationDialog;
                loader.item.loadStation(station);
                loader.item.open();
            }
        }

        MyToolButton {
            iconSource: "toolbar-delete"
            toolTipText: qsTr("Delete")
            onClicked: {
                loader.sourceComponent = confirmDeleteDialog;
                loader.item.open();
            }
        }
    }

    Station {
        id: station
    }

    MyFlickable {
        id: flicker

        anchors {
            fill: parent
            margins: platformStyle.paddingLarge
        }
        contentHeight: column.height + platformStyle.paddingLarge

        Column {
            id: column

            anchors {
                top: parent.top
                left: parent.left
                right: parent.right
            }
            spacing: platformStyle.paddingLarge

            Logo {
                width: 120
                height: 120
                image: station.logo
                text: station.title
                enabled: (!MediaPlayer.currentStation) || (MediaPlayer.currentStation.id != station.id)
                onClicked: Settings.mediaPlayer == "cuteradio" ? MediaPlayer.play(station)
                                                               : Utils.urlIsPlaylist(station.source)
                                                                 ? StreamExtractor.getStreamUrl(station.source)
                                                                 : Launcher.launch(station.source)
            }

            Label {
                width: parent.width
                wrapMode: Text.WordWrap
                text: station.description
            }

            SeparatorLabel {
                width: parent.width
                text: qsTr("Properties")
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Genre") + ": " + station.genre
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Country") + ": " + station.country
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Language") + ": " + station.language
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Source") + ": " + station.source
            }

            Label {
                width: parent.width
                elide: Text.ElideRight
                font.pixelSize: platformStyle.fontSizeSmall
                text: qsTr("Last played") + ": " + Utils.dateFromMSecs(station.lastPlayed)
                visible: station.lastPlayed > 0
            }
        }
    }

    MyScrollBar {
        flickableItem: flicker
    }

    Loader {
        id: loader
    }

    Component {
        id: editStationDialog

        AddStationDialog {}
    }

    Component {
        id: confirmDeleteDialog

        MyQueryDialog {
            titleText: qsTr("Delete station?")
            icon: "images/info.png"
            message: qsTr("Do you wish to delete the station") + " '" + station.title + "' " + qsTr("from the database?")
            acceptButtonText: qsTr("Yes")
            rejectButtonText: qsTr("No")
            onAccepted: Database.asyncDeleteStation(station.id)
            onStatusChanged: if ((status === DialogStatus.Closed) && (root.status === PageStatus.Active)) flicker.forceActiveFocus();
        }
    }

    Connections {
        target: Database
        onStationDeleted: if ((id === station.id) && (status === PageStatus.Active)) appWindow.pageStack.pop();
    }
}
