/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

import QtQuick 1.1
import com.nokia.meego 1.0
import com.cuteradio.items 1.0
import "file:///usr/lib/qt4/imports/com/nokia/meego/UIConstants.js" as UI

MySheet {
    id: root

    function loadPodcast(podcastItem) {
        podcast.loadStation(podcastItem);
    }

    function addPodcast() {
        Database.asyncAddPodcast(titleField.text,
                            !descriptionField.text ? qsTr("No description") : descriptionField.text,
                            !logoField.text ? "" : downloader.localPath,
                            sourceField.text);
        clear();
    }

    function updatePodcast() {
        var properties = {};

        if (titleField.text != podcast.title) {
            properties["title"] = titleField.text;
        }
        if (descriptionField.text != podcast.description) {
            properties["description"] = !descriptionField.text ? qsTr("No description") : descriptionField.text;
        }
        if (logoField.text != podcast.logo) {
            properties["logo"] = !logoField.text ? "" : downloader.localPath;
        }
        if (sourceField.text != podcast.source) {
            properties["source"] = sourceField.text;
        }

        Database.asyncUpdatePodcast(podcast.id, properties);

        clear();
    }

    function clear() {
        if (podcast.id) {
            podcast.clear();
        }
        else {
            titleField.text = "";
            descriptionField.text = "";
            logoField.text = "";
            sourceField.text = "";
        }
    }

    Station {
        id: podcast
    }

    LogoDownloader {
        id: downloader

        url: logoField.text
        localPath: Utils.logoPath + (podcast.id ? podcast.id : podcast.title) + ".png"
        onFinished: podcast.id ? updatePodcast() : addPodcast()
    }

    acceptButtonText: (titleField.text) && (sourceField.text) ? qsTr("Done") : ""
    rejectButtonText: qsTr("Cancel")
    content: Item {
        anchors.fill: parent
        clip: true

        Flickable {
            id: flicker

            anchors {
                fill: parent
                margins: UI.PADDING_DOUBLE
            }
            contentHeight: column.height + UI.PADDING_DOUBLE

            Column {
                id: column

                anchors {
                    top: parent.top
                    left: parent.left
                    right: parent.right
                }
                spacing: UI.PADDING_DOUBLE

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Title")
                }

                MyTextField {
                    id: titleField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: podcast.title
                    onAccepted: descriptionField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Description")
                }

                MyTextArea {
                    id: descriptionField

                    width: parent.width
                    text: podcast.description
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Logo")
                }

                MyTextField {
                    id: logoField

                    width: parent.width
                    actionLabel: qsTr("Next")
                    text: podcast.logo
                    onAccepted: sourceField.forceActiveFocus()
                }

                Label {
                    width: parent.width
                    elide: Text.ElideRight
                    font.bold: true
                    text: qsTr("Source")
                }

                MyTextField {
                    id: sourceField

                    width: parent.width
                    inputMethodHints: Qt.ImhNoAutoUppercase | Qt.ImhUrlCharactersOnly
                    text: podcast.source
                    onAccepted: platformCloseSoftwareInputPanel()
                }
            }
        }

        ScrollDecorator {
            flickableItem: flicker
        }
    }

    onAccepted: downloader.localPath.substring(0, 4) == "file" ? station.id ? updatePodcast() : addPodcast() : downloader.start()
    onRejected: clear()
}
