/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QStringList>
#include <QTimerEvent>
#include "mediaplayer.h"
#include "../../base/streamextractor.h"
#include "../../base/utils.h"
#include "../../base/database.h"
#include "../../base/settings.h"

#define SLEEP_TIMER_INTERVAL 60000

MediaPlayer* MediaPlayer::self = 0;

MediaPlayer::MediaPlayer(QObject *parent) :
    QObject(parent),
    m_mediaPlayer(new Phonon::MediaObject(this)),
    m_audioOutput(new Phonon::AudioOutput(Phonon::MusicCategory, this)),
    m_extractor(new StreamExtractor(this)),
    m_station(0),
    m_state(MediaState::Stopped),
    m_playbackMode(PlaybackMode::Radio),
    m_bufferProgress(0),
    m_sleepTimerEnabled(false),
    m_sleepTimerRemaining(60000),
    m_sleepTimerId(0)
{
    if (!self) {
        self = this;
    }

    Phonon::createPath(m_mediaPlayer, m_audioOutput);

    this->connect(m_mediaPlayer, SIGNAL(tick(qint64)), this, SIGNAL(positionChanged(qint64)));
    this->connect(m_mediaPlayer, SIGNAL(totalTimeChanged(qint64)), this, SIGNAL(durationChanged(qint64)));
    this->connect(m_mediaPlayer, SIGNAL(bufferStatus(int)), this, SLOT(onBufferProgressChanged(int)));
    this->connect(m_mediaPlayer, SIGNAL(stateChanged(Phonon::State, Phonon::State)), this, SLOT(onStateChanged(Phonon::State)));
    this->connect(m_mediaPlayer, SIGNAL(metaDataChanged()), this, SIGNAL(metaDataChanged()));
    this->connect(m_audioOutput, SIGNAL(volumeChanged(qreal)), this, SIGNAL(volumeChanged(qreal)));
    this->connect(m_extractor, SIGNAL(gotStreamUrl(QUrl)), this, SLOT(play(QUrl)));
    this->connect(m_extractor, SIGNAL(error(QString)), this, SLOT(onExtractorError(QString)));

#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet = new ResourcePolicy::ResourceSet("player", this);
    m_audioResource = new ResourcePolicy::AudioResource("player");
    m_audioResource->setProcessID(QCoreApplication::applicationPid());
    m_audioResource->setStreamTag("media.name", "*");
    m_audioResource->setOptional(false);
    m_resourceSet->addResourceObject(m_audioResource);

    this->connect(m_resourceSet, SIGNAL(resourcesGranted(QList<ResourcePolicy::ResourceType>)), this, SLOT(onResourcesGranted()));
    this->connect(m_resourceSet, SIGNAL(resourcesReleased()), this, SLOT(onResourcesReleased()));
    this->connect(m_resourceSet, SIGNAL(resourcesReleasedByManager()), this, SLOT(onResourcesReleased()));
    this->connect(m_resourceSet, SIGNAL(resourcesDenied()), this, SLOT(onResourcesDenied()));
    this->connect(m_resourceSet, SIGNAL(lostResources()), this, SLOT(onResourcesLost()));
#endif
}

MediaPlayer::~MediaPlayer() {
#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet->release();
#endif
}

MediaPlayer* MediaPlayer::instance() {
    return !self ? new MediaPlayer : self;
}

Station* MediaPlayer::currentStation() const {
    return m_station;
}

void MediaPlayer::setCurrentStation(Station *station) {
    if (m_station) {
        m_station->deleteLater();
        m_station = 0;
    }

    if (station) {
        m_station = new Station(station, this);
    }

    emit currentStationChanged(this->currentStation());
}

PlaybackMode::Mode MediaPlayer::playbackMode() const {
    return m_playbackMode;
}

void MediaPlayer::setPlaybackMode(PlaybackMode::Mode playbackMode) {
    if (playbackMode != this->playbackMode()) {
        m_playbackMode = playbackMode;
        emit playbackModeChanged(playbackMode);
    }
}

MediaState::State MediaPlayer::state() const {
    return m_state;
}

void MediaPlayer::setState(MediaState::State state) {
    if (state != this->state()) {
        m_state = state;
        emit stateChanged(state);
    }
}

QString MediaPlayer::stateString() const {
    if (this->buffering()) {
        return tr("Buffering media");
    }

    switch (this->state()) {
    case Phonon::PlayingState:
        return tr("Playing");
    case Phonon::PausedState:
        return tr("Paused");
    case Phonon::StoppedState:
        return tr("Stopped");
    case Phonon::ErrorState:
        return tr("Error");
    default:
        return tr("Unknown");
    }
}

bool MediaPlayer::playing() const {
    switch (this->state()) {
    case MediaState::Playing:
    case MediaState::Buffering:
        return true;
    default:
        return false;
    }
}

bool MediaPlayer::paused() const {
    return this->state() == MediaState::Paused;
}

bool MediaPlayer::buffering() const {
    return this->state() == MediaState::Buffering;
}

QString MediaPlayer::errorString() const {
    return m_errorString;
}

void MediaPlayer::setErrorString(const QString &errorString) {
    m_errorString = errorString;
}

qint64 MediaPlayer::position() const {
    return m_mediaPlayer->currentTime();
}

void MediaPlayer::setPosition(qint64 position) {
    m_mediaPlayer->seek(position);
}

qint64 MediaPlayer::duration() const {
    return m_mediaPlayer->totalTime();
}

int MediaPlayer::bufferProgress() const {
    return m_bufferProgress;
}

qreal MediaPlayer::volume() const {
    return m_audioOutput->volume();
}

void MediaPlayer::setVolume(qreal volume) {
    m_audioOutput->setVolume(volume);
}

QString MediaPlayer::title() const {
    return m_mediaPlayer->metaData(Phonon::TitleMetaData).isEmpty() ? QString()
                                                                    : m_mediaPlayer->metaData(Phonon::TitleMetaData).first();
}

QString MediaPlayer::artist() const {
    return m_mediaPlayer->metaData(Phonon::ArtistMetaData).isEmpty() ? QString()
                                                                     : m_mediaPlayer->metaData(Phonon::ArtistMetaData).first();
}

QString MediaPlayer::album() const {
    return m_mediaPlayer->metaData(Phonon::AlbumMetaData).isEmpty() ? QString()
                                                                    : m_mediaPlayer->metaData(Phonon::AlbumMetaData).first();
}

QString MediaPlayer::genre() const {
    return m_mediaPlayer->metaData(Phonon::GenreMetaData).isEmpty() ? QString()
                                                                    : m_mediaPlayer->metaData(Phonon::GenreMetaData).first();
}

QString MediaPlayer::description() const {
    return m_mediaPlayer->metaData(Phonon::DescriptionMetaData).isEmpty() ? QString()
                                                                          : m_mediaPlayer->metaData(Phonon::DescriptionMetaData).first();
}

qint64 MediaPlayer::bitRate() const {
    return m_mediaPlayer->metaData("BITRATE").isEmpty() ? 0
                                                        : m_mediaPlayer->metaData("BITRATE").first().toLongLong();
}

bool MediaPlayer::sleepTimerEnabled() const {
    return m_sleepTimerEnabled;
}

void MediaPlayer::setSleepTimerEnabled(bool enabled) {
    if (enabled != this->sleepTimerEnabled()) {
        m_sleepTimerEnabled = enabled;
        emit sleepTimerEnabledChanged(enabled);

        if (enabled) {
            int duration = Settings::instance()->sleepTimerDuration();
            this->setSleepTimerRemaining(!duration ? SLEEP_TIMER_INTERVAL : duration * SLEEP_TIMER_INTERVAL);
            this->setSleepTimerId(this->startTimer(SLEEP_TIMER_INTERVAL));
        }
        else {
            this->killTimer(this->sleepTimerId());
        }
    }
}

qint64 MediaPlayer::sleepTimerRemaining() const {
    return m_sleepTimerRemaining;
}

void MediaPlayer::setSleepTimerRemaining(qint64 remaining) {
    if (remaining != this->sleepTimerRemaining()) {
        m_sleepTimerRemaining = remaining;
        emit sleepTimerEnabledChanged(remaining);
    }
}

qint64 MediaPlayer::tickInterval() const {
    return m_mediaPlayer->tickInterval();
}

void MediaPlayer::setTickInterval(qint64 interval) {
    m_mediaPlayer->setTickInterval(interval);
}

void MediaPlayer::play() {
    switch (this->state()) {
    case MediaState::Error:
        this->play(this->currentStation() ? this->currentStation()->source()
                                          : m_mediaPlayer->currentSource().url());
        return;
    default:
        break;
    }
#ifdef MEEGO_EDITION_HARMATTAN
    m_resourceSet->acquire();
#else
    m_mediaPlayer->play();
#endif
}

void MediaPlayer::play(const QUrl &url) {
    switch (this->state()) {
    case MediaState::Stopped:
        break;
    default:
        this->stop();
        break;
    }

    if (url.scheme() == "file") {
        // url is a local file
        this->setPlaybackMode(PlaybackMode::Sequential);
        m_mediaPlayer->setCurrentSource(url);
        this->play();
    }
    else {
        this->setPlaybackMode(PlaybackMode::Radio);

        if (Utils::urlIsPlaylist(url)) {
            m_extractor->getStreamUrl(url);
        }
        else {
            m_mediaPlayer->setCurrentSource(url);
            this->play();
        }
    }
}

void MediaPlayer::play(Station *station) {
    this->setCurrentStation(station);
    this->play(station->source());
}

void MediaPlayer::pause() {
    m_mediaPlayer->pause();
}

void MediaPlayer::stop() {
    m_mediaPlayer->stop();
}

void MediaPlayer::clear() {
    this->stop();
    this->setCurrentStation(0);
}

void MediaPlayer::updateSleepTimer() {
    this->setSleepTimerRemaining(this->sleepTimerRemaining() - SLEEP_TIMER_INTERVAL);

    if (!this->sleepTimerRemaining()) {
        this->stop();
    }
}

int MediaPlayer::sleepTimerId() const {
    return m_sleepTimerId;
}

void MediaPlayer::setSleepTimerId(int timerId) {
    m_sleepTimerId = timerId;
}

void MediaPlayer::timerEvent(QTimerEvent *event) {
    if (!this->sleepTimerEnabled()) {
        this->killTimer(event->timerId());
    }
    else {
        this->updateSleepTimer();
    }
}

void MediaPlayer::onBufferProgressChanged(int progress) {
    m_bufferProgress = progress;
    emit bufferProgressChanged(progress);
}

void MediaPlayer::onStateChanged(Phonon::State state) {
    switch (state) {
    case Phonon::BufferingState:
        this->setState(MediaState::Buffering);
        break;
    case Phonon::PlayingState:
        this->setState(MediaState::Playing);
        this->onStationPlayed();
        break;
    case Phonon::PausedState:
        this->setState(MediaState::Paused);
    case Phonon::StoppedState:
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        this->setState(MediaState::Stopped);
        this->setSleepTimerEnabled(false);
        break;
    case Phonon::ErrorState:
#ifdef MEEGO_EDITION_HARMATTAN
        m_resourceSet->release();
#endif
        this->setErrorString(m_mediaPlayer->errorString());
        this->setState(MediaState::Error);
        this->setSleepTimerEnabled(false);
        emit error(this->errorString());
        break;
    default:
        break;
    }
}

void MediaPlayer::onExtractorError(const QString &errorString) {
    this->setErrorString(errorString);
    this->setState(MediaState::Error);
    emit error(errorString);
}

void MediaPlayer::onStationPlayed() {
    if (this->currentStation()) {
        Database::asyncUpdateStation(this->currentStation()->id(), "lastPlayed", QDateTime::currentMSecsSinceEpoch(), false);
    }
}

#ifdef MEEGO_EDITION_HARMATTAN
void MediaPlayer::onResourcesGranted() {
    m_mediaPlayer->play();
}

void MediaPlayer::onResourcesReleased() {
    qDebug() << "Resources released";
}

void MediaPlayer::onResourcesDenied() {
    qDebug() << "Resources denied";
}

void MediaPlayer::onResourcesLost() {
    qDebug() << "Resources lost";
}
#endif
