/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <QtSql>
#include <QCoreApplication>
#include <QThread>
#include <QDir>
#include <qplatformdefs.h>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif
#include "database.h"

Database* Database::self = 0;

Database::Database(QObject *parent) :
    QObject(parent)
{
    if (!self) {
        self = this;
    }

    this->moveToThread(new QThread);
    this->thread()->start();

    QDir dir;
#ifdef Q_OS_SYMBIAN
    dir.setPath(QCoreApplication::applicationDirPath());
    dir.cd("stations");
#elif (defined MEEGO_EDITION_HARMATTAN) || (defined Q_WS_MAEMO_5)
    dir.setPath("/home/user/cuteRadio/");
#elif QT_VERSION >= 0x050000
    dir.setPath(QStandardPaths::writeableLocation(QStandardPaths::HomeLocation));
    dir.cd("cuteRadio");
#else
    dir.setPath(QDesktopServices::storageLocation(QDesktopServices::HomeLocation));
    dir.cd("cuteRadio");
#endif
    QSqlDatabase database = QSqlDatabase::addDatabase("QSQLITE");
    database.setDatabaseName(QDir::toNativeSeparators(dir.path() + "/cuteradio.db"));
}

Database::~Database() {
    this->thread()->quit();
    this->thread()->deleteLater();
}

Database* Database::instance() {
    return !self ? new Database : self;
}

bool Database::addStation(const QString &title,
                          const QString &description,
                          const QString &genre,
                          const QString &country,
                          const QString &language,
                          const QUrl &source,
                          bool favourite) const {

    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;    
    query.prepare("INSERT INTO stations VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
    qsrand(QDateTime::currentMSecsSinceEpoch());
    int id = qrand();
    query.addBindValue(id);
    query.addBindValue(title);
    query.addBindValue(description);
    query.addBindValue(genre);
    query.addBindValue(country);
    query.addBindValue(language);
    query.addBindValue(source);
    query.addBindValue(favourite ? 1 : 0);
    query.addBindValue(0);

    if (query.exec()) {
        this->addGenre(genre);
        this->addCountry(country);
        this->addLanguage(language);

        emit stationAdded();
        emit alert(QString("%1 '%2' %3").arg(tr("New station")).arg(title).arg(tr("added")));

        return true;
    }

    emit error(tr("Cannot add new station"));

    return false;
}

bool Database::updateStation(const QString &id, const QVariantMap &properties, bool notify) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    QString oldGenre;
    QString oldCountry;
    QString oldLanguage;

    if ((properties.contains("genre")) || (properties.contains("country")) || (properties.contains("language"))) {
        query.prepare("SELECT genre, country, language FROM stations WHERE id = ?");
        query.addBindValue(id);

        if (query.exec()) {

            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    oldGenre = query.value(0).toString();
                    oldCountry = query.value(1).toString();
                    oldLanguage = query.value(2).toString();
                }
            }
        }
    }

    QString queryString("UPDATE stations SET ");
    QMapIterator<QString, QVariant>iterator(properties);

    while (iterator.hasNext()) {
        iterator.next();
        queryString.append(iterator.hasNext() ? QString("%1 = '%2', ").arg(iterator.key()).arg(iterator.value().toString())
                                              : QString("%1 = '%2' ").arg(iterator.key()).arg(iterator.value().toString()));

    }

    queryString.append("WHERE id = " + id);

    if (query.exec(queryString)) {
        if (properties.contains("genre")) {
            this->updateGenreStationCount(oldGenre);
            this->addGenre(properties.value("genre").toString());
        }
        if (properties.contains("country")) {
            this->updateCountryStationCount(oldCountry);
            this->addCountry(properties.value("country").toString());
        }
        if (properties.contains("language")) {
            this->updateLanguageStationCount(oldLanguage);
            this->addLanguage(properties.value("language").toString());
        }

        emit stationUpdated(id, properties);

        if (notify) {
            emit alert(tr("Station updated"));
        }

        return true;
    }

    if (notify) {
        emit error(tr("Cannot update station"));
    }

    return false;
}

bool Database::updateStation(const QString &id, const QString &property, const QVariant &value, bool notify) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    QString oldValue;

    if ((property == "genre") || (property == "country") || (property == "language")) {
        query.prepare("SELECT ? FROM stations WHERE id = ?");
        query.addBindValue(property);
        query.addBindValue(id);

        if (query.exec()) {

            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    oldValue = query.value(0).toString();
                }
            }
        }
    }

    query.prepare(QString("UPDATE stations SET %1 = ? WHERE id = ?").arg(property));
    query.addBindValue(value);
    query.addBindValue(id);

    if (query.exec()) {
        if (property == "genre") {
            this->updateGenreStationCount(oldValue);
            this->addGenre(value.toString());
        }
        else if (property == "country") {
            this->updateCountryStationCount(oldValue);
            this->addCountry(value.toString());
        }
        else if (property == "language") {
            this->updateLanguageStationCount(oldValue);
            this->addLanguage(value.toString());
        }

        emit stationUpdated(id, property, value);

        if (notify) {
            emit alert(tr("Station updated"));
        }

        return true;
    }

    if (notify) {
        emit error(tr("Cannot update station"));
    }

    return false;
}

bool Database::deleteStation(const QString &id) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("SELECT genre, country, language FROM stations WHERE id = ?");
    query.addBindValue(id);

    if (query.exec()) {
        QString genre;
        QString country;
        QString language;
        QSqlRecord record = query.record();

        if (record.count()) {
            while (query.next()) {
                genre = query.value(0).toString();
                country = query.value(1).toString();
                language = query.value(2).toString();
            }
        }

        query.prepare("DELETE FROM stations WHERE id = ?");
        query.addBindValue(id);

        if (query.exec()) {
            this->updateGenreStationCount(genre);
            this->updateCountryStationCount(country);
            this->updateLanguageStationCount(language);

            emit stationDeleted(id);
            emit alert(tr("Station deleted"));

            return true;
        }

        emit error(tr("Cannot delete station"));

        return false;
    }

    emit error(tr("Cannot delete station"));

    return false;
}

bool Database::asyncAddStation(const QString &title,
                               const QString &description,
                               const QString &genre,
                               const QString &country,
                               const QString &language,
                               const QUrl &source,
                               bool favourite) {

    return QMetaObject::invokeMethod(Database::instance(), "addStation", Qt::QueuedConnection,
                                     Q_ARG(QString, title),
                                     Q_ARG(QString, description),
                                     Q_ARG(QString, genre),
                                     Q_ARG(QString, country),
                                     Q_ARG(QString, language),
                                     Q_ARG(QUrl, source),
                                     Q_ARG(bool, favourite));
}

bool Database::asyncUpdateStation(const QString &id, const QVariantMap &properties, bool notify) {
    return QMetaObject::invokeMethod(Database::instance(), "updateStation", Qt::QueuedConnection,
                                     Q_ARG(QString, id), Q_ARG(QVariantMap, properties), Q_ARG(bool, notify));
}

bool Database::asyncUpdateStation(const QString &id, const QString &property, const QVariant &value, bool notify) {
    return QMetaObject::invokeMethod(Database::instance(), "updateStation", Qt::QueuedConnection,
                                     Q_ARG(QString, id), Q_ARG(QString, property), Q_ARG(QVariant, value), Q_ARG(bool, notify));
}

bool Database::asyncDeleteStation(const QString &id) {
    return QMetaObject::invokeMethod(Database::instance(), "deleteStation", Qt::QueuedConnection, Q_ARG(QString, id));
}

bool Database::addGenre(const QString &genre) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    int count = this->getStationCount("genre", genre);

    if (count == 1) {
        query.prepare("INSERT INTO genres VALUES (?, ?)");
        query.addBindValue(genre);
        query.addBindValue(count);

        if (query.exec()) {
            emit genreAdded(genre, count);

            return true;
        }

        return false;
    }
    else {
        query.prepare("UPDATE genres SET stationCount = ? WHERE name = ?");
        query.addBindValue(count);
        query.addBindValue(genre);

        if (query.exec()) {
            emit genreUpdated(genre, count);

            return true;
        }

        return false;
    }
}

bool Database::updateGenreStationCount(const QString &genre) const {
    int count = this->getStationCount("genre", genre);

    if (!count) {
        return this->deleteGenre(genre);
    }

    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("UPDATE genres SET stationCount = ? WHERE name = ?");
    query.addBindValue(count);
    query.addBindValue(genre);

    if (query.exec()) {
        emit genreUpdated(genre, count);

        return true;
    }

    return false;
}

bool Database::deleteGenre(const QString &genre) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("DELETE FROM genres WHERE name = ?");
    query.addBindValue(genre);

    if (query.exec()) {
        emit genreDeleted(genre);

        return true;
    }

    return false;
}

bool Database::addCountry(const QString &country) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    int count = this->getStationCount("country", country);

    if (count == 1) {
        query.prepare("INSERT INTO countries VALUES (?, ?)");
        query.addBindValue(country);
        query.addBindValue(count);

        if (query.exec()) {
            emit countryAdded(country, count);

            return true;
        }

        return false;
    }
    else {
        query.prepare("UPDATE countries SET stationCount = ? WHERE name = ?");
        query.addBindValue(count);
        query.addBindValue(country);

        if (query.exec()) {
            emit countryUpdated(country, count);

            return true;
        }

        return false;
    }
}

bool Database::updateCountryStationCount(const QString &country) const {
    int count = this->getStationCount("country", country);

    if (!count) {
        return this->deleteCountry(country);
    }

    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("UPDATE countries SET stationCount = ? WHERE name = ?");
    query.addBindValue(count);
    query.addBindValue(country);

    if (query.exec()) {
        emit countryUpdated(country, count);

        return true;
    }

    return false;
}

bool Database::deleteCountry(const QString &country) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("DELETE FROM countries WHERE name = ?");
    query.addBindValue(country);

    if (query.exec()) {
        emit countryDeleted(country);

        return true;
    }

    return false;
}

bool Database::addLanguage(const QString &language) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    int count = this->getStationCount("language", language);

    if (count == 1) {
        query.prepare("INSERT INTO languages VALUES (?, ?)");
        query.addBindValue(language);
        query.addBindValue(count);

        if (query.exec()) {
            emit languageAdded(language, count);

            return true;
        }

        return false;
    }
    else {
        query.prepare("UPDATE languages SET stationCount = ? WHERE name = ?");
        query.addBindValue(count);
        query.addBindValue(language);

        if (query.exec()) {
            emit languageUpdated(language, count);

            return true;
        }

        return false;
    }
}

bool Database::updateLanguageStationCount(const QString &language) const {
    int count = this->getStationCount("language", language);

    if (!count) {
        return this->deleteLanguage(language);
    }

    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("UPDATE languages SET stationCount = ? WHERE name = ?");
    query.addBindValue(count);
    query.addBindValue(language);

    if (query.exec()) {
        emit languageUpdated(language, count);

        return true;
    }

    return false;
}

bool Database::deleteLanguage(const QString &language) const {
    if (!QSqlDatabase::database().open()) {
        return false;
    }

    QSqlQuery query;
    query.prepare("DELETE FROM langauges WHERE name = ?");
    query.addBindValue(language);

    if (query.exec()) {
        emit languageDeleted(language);

        return true;
    }

    return false;
}

int Database::getStationCount(const QString &property, const QVariant &value) const {
    int count = 0;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare(QString("SELECT COUNT (*) FROM stations WHERE %1 = ?").arg(property));
        query.addBindValue(value);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    count = query.value(0).toInt();
                }
            }
        }
    }

    return count;
}

QList<Station*> Database::getRecentlyPlayedStations() const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE lastPlayed > 0 ORDER BY lastPlayed DESC");

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getFavouriteStations() const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE favourite = 1 ORDER BY title ASC");

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsById(const QStringList &ids) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        QString queryString("SELECT * FROM stations WHERE id = ");

        for (int i = 0; i < ids.size(); i++) {
            queryString.append(i < (ids.size() - 1) ? ids.at(i) + " OR id = " : ids.at(i));
        }

        if (query.exec(queryString)) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByTitle(const QString &title) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec(QString("SELECT * FROM stations WHERE title LIKE '%%1%' OR description LIKE '%%1%' ORDER BY title ASC").arg(title))) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByGenre(const QString &genre) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE genre = ? ORDER BY title ASC");
        query.addBindValue(genre);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByCountry(const QString &country) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE country = ? ORDER BY title ASC");
        query.addBindValue(country);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

QList<Station*> Database::getStationsByLanguage(const QString &language) const {
    QList<Station*> stations;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;
        query.prepare("SELECT * FROM stations WHERE language = ? ORDER BY title ASC");
        query.addBindValue(language);

        if (query.exec()) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    Station *station = new Station(query.value(0).toString(),
                                                   query.value(1).toString(),
                                                   query.value(2).toString(),
                                                   query.value(3).toString(),
                                                   query.value(4).toString(),
                                                   query.value(5).toString(),
                                                   query.value(6).toUrl(),
                                                   query.value(7).toBool(),
                                                   query.value(8).toLongLong());

                    stations.append(station);
                }
            }
        }
    }

    return stations;
}

void Database::asyncGetRecentlyPlayedStations() const {
    emit gotStations(this->getRecentlyPlayedStations());
}

void Database::asyncGetFavouriteStations() const {
    emit gotStations(this->getFavouriteStations());
}

void Database::asyncGetStationsById(const QStringList &ids) const {
    emit gotStations(this->getStationsById(ids));
}

void Database::asyncGetStationsByTitle(const QString &title) const {
    emit gotStations(this->getStationsByTitle(title));
}

void Database::asyncGetStationsByGenre(const QString &genre) const {
    emit gotStations(this->getStationsByGenre(genre));
}

void Database::asyncGetStationsByCountry(const QString &country) const {
    emit gotStations(this->getStationsByCountry(country));
}

void Database::asyncGetStationsByLanguage(const QString &language) const {
    emit gotStations(this->getStationsByLanguage(language));
}

QList< QPair<QString, int> > Database::getGenres() const {
    QList< QPair<QString, int> > genres;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec("SELECT * FROM genres ORDER BY name ASC")) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    genres.append(QPair<QString, int>(query.value(0).toString(), query.value(1).toInt()));
                }
            }
        }
    }

    return genres;
}

QList< QPair<QString, int> > Database::getCountries() const {
    QList< QPair<QString, int> > countries;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec("SELECT * FROM countries ORDER BY name ASC")) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    countries.append(QPair<QString, int>(query.value(0).toString(), query.value(1).toInt()));
                }
            }
        }
    }

    return countries;
}

QList< QPair<QString, int> > Database::getLanguages() const {
    QList< QPair<QString, int> > languages;

    if (QSqlDatabase::database().open()) {
        QSqlQuery query;

        if (query.exec("SELECT * FROM languages ORDER BY name ASC")) {
            QSqlRecord record = query.record();

            if (record.count()) {
                while (query.next()) {
                    languages.append(QPair<QString, int>(query.value(0).toString(), query.value(1).toInt()));
                }
            }
        }
    }

    return languages;
}

void Database::asyncGetGenres() const {
    emit gotGenres(this->getGenres());
}

void Database::asyncGetCountries() const {
    emit gotCountries(this->getCountries());
}

void Database::asyncGetLanguages() const {
    emit gotLanguages(this->getLanguages());
}
