/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */
#ifndef STATIONLISTMODEL_H
#define STATIONLISTMODEL_H

#include <QAbstractListModel>
#include "station.h"
#include "enums.h"

class StationListModel : public QAbstractListModel
{
    Q_OBJECT
    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(QString searchQuery
               READ searchQuery
               NOTIFY searchQueryChanged)
    Q_PROPERTY(Queries::QueryType queryType
               READ queryType
               NOTIFY queryTypeChanged)

public:
    enum Roles {
        IdRole = Qt::UserRole + 1,
        TitleRole,
        DescriptionRole,
        GenreRole,
        CountryRole,
        LanguageRole,
        SourceRole,
        FavouriteRole,
        LastPlayedRole,
        SectionRole
    };

    explicit StationListModel(QObject *parent = 0);
    ~StationListModel();

#if QT_VERSION >= 0x050000
    QHash<int, QByteArray> roleNames() const;
#endif

    int rowCount(const QModelIndex &parent = QModelIndex()) const;

    QVariant data(const QModelIndex &index, int role) const;
    Q_INVOKABLE QVariant data(int row, const QByteArray &role) const;
    bool setData(const QModelIndex &index, const QVariant &value, int role);
    Q_INVOKABLE bool setData(int row, const QVariant &value, const QByteArray &role);

    bool loading() const;
    QString searchQuery() const;
    Queries::QueryType queryType() const;

    Station* get(const QModelIndex &index) const;
    Q_INVOKABLE Station* get(int row) const;

public slots:
    void searchStations(const QString &query) const;
    void showFavouriteStations() const;
    void showRecentlyPlayedStations() const;
    void showStationsByGenre(const QString &genre) const;
    void showStationsByCountry(const QString &country) const;
    void showStationsByLanguage(const QString &language) const;
    void clear();
    void reload();

protected:
    void setLoading(bool loading) const;
    void setSearchQuery(const QString &query) const;
    void setQueryType(Queries::QueryType queryType) const;

protected slots:
    void addStations(const QList<Station*> &stations);
    void addStation(Station *station);
    void insertStation(int row, Station *station);
    void removeStation(int row);
    void removeStation(int role, const QVariant &value);
    void onStationUpdated(const QString &id, const QVariantMap &properties);
    void onStationUpdated(const QString &id, const QString &property, const QVariant &value);
    void onStationDeleted(const QString &id);

signals:
    void countChanged(int count) const;
    void loadingChanged(bool loading) const;
    void searchQueryChanged(const QString &query) const;
    void queryTypeChanged(Queries::QueryType queryType) const;

private:
    QList<Station*> m_list;
    mutable bool m_loading;
    mutable QString m_searchQuery;
    mutable Queries::QueryType m_queryType;
    QHash<int, QByteArray> m_roleNames;
};

#endif // STATIONLISTMODEL_H
