#include "util.h"
#include <QtGlobal>
#include <math.h>


using namespace std;

#define BUFFER_SIZE 4096

/* Function to keep waiting for data until the amount expected has been received, or until timeout occurs.
buffer must point to a buffer with enough space to fit size bytes */
bool RecvData(QTcpSocket * sock, char * buffer, int size, int timeout){
    int total_bytes_received = 0;
    int recvlimit = min(BUFFER_SIZE, size); //limits the amount of data we recv from the socket per iteration
    while(1){
        char recvbuf[BUFFER_SIZE] = {0};
        int bytes_received = sock->read(recvbuf, recvlimit);
        bytes_received = max(bytes_received, 0);
        memcpy(buffer + total_bytes_received, recvbuf, bytes_received);
        total_bytes_received += bytes_received;
        if(total_bytes_received == size){
            return true;
        }else if (total_bytes_received > size){
            //Received an excess of data
            return false;
        }
    }
}

int LangToInt(QString lang){
    if(lang == "ENG"){
        return 1;
    }else if(lang == "CHI-t"){
        return 2;
    }else if(lang == "CHI-s"){
        return 3;
    }else if(lang == "JPN"){
        return 4;
    }else if(lang == "KOR"){
        return 5;
    }
    return 0;
}

QString IntToLang(int i){
    switch(i){
    case 1:
        return "eng";
    case 2:
        return "chi_tra";
    case 3:
        return "chi_sim";
    case 4:
        return "jpn";
    case 5:
        return "kor";
    }
    return 0;
}

// Write the file to the socket out buffer
bool SendFileData(QTcpSocket * sock, QString filename){

    QFile file(filename);
    file.open(QIODevice::ReadOnly);
    int error = file.error();
    QDataStream fs(&file);
    int status = fs.status();
    int total_sent = 0;
    while(1){
        char buffer[BUFFER_SIZE] = {0};
        int bytes_read = fs.readRawData(buffer, sizeof(buffer));
        total_sent += bytes_read;
        char * p = buffer;
        while(bytes_read > 0){
            int bytes_sent = sock->write(p, bytes_read);
            bytes_read -= bytes_sent;
            p += bytes_sent;
        }

        if(fs.atEnd()){
            /* Done sending the results*/
            file.close();
            return true;
        }
    }
    file.close();
    return false;
}

// Write the buffer of length bytes to the socket out buffer
bool SendBufferData(QTcpSocket * sock, char * buffer, int length){
    int total_bytes_sent = 0;
    while(total_bytes_sent < length){
        int bytes_sent = sock->write(buffer + total_bytes_sent, length - total_bytes_sent);
        total_bytes_sent += bytes_sent;
    }
    return true;
}

void WriteToFile(char * data, char * filename, int length){
    FILE * file = fopen(filename, "wb");
    QFile qfile;
    qfile.open(file, QIODevice::WriteOnly | QIODevice::Append);
    int total_written = 0;
    while (total_written != length){
        int bytes_written = qfile.write(data + total_written, length - total_written);
        total_written += bytes_written;
    }
    qfile.close();
    fclose(file);
}

QImage GloballyThreshold(QImage in, int t){
    QImage thresolded(in.width(), in.height(), QImage::Format_Mono);
    for(int i = 0; i < in.width(); i++){
        for(int j = 0; j < in.height(); j++){
            unsigned int pixel = in.pixel(i, j);
            pixel = qGray(pixel);
            if(pixel >= t){
                // White pixel
                thresolded.setPixel(i,j,1);
            }else{
                // Black pixel
                thresolded.setPixel(i,j,0);
            }
        }
    }
    return thresolded;
}

void AdaptiveThresholder::Init(QImage img){
    srcImg = img;
    integralImg.clear();
    squaredIntegralImg.clear();
    integralImg = ComputeIntegralImage(false);
    squaredIntegralImg = ComputeIntegralImage(true);
}

QVector<QVector<quint64> > AdaptiveThresholder::ComputeIntegralImage(bool squared){
    QVector<QVector<quint64> > rtn;
    for(int i = 0; i < srcImg.width(); i++){ // i refers to the column
        QVector<quint64> column;
        for(int j = 0; j < srcImg.height(); j++){ // j refers to the row
            quint64 sum = 0;
            unsigned int pixel = srcImg.pixel(i, j);
            pixel = qGray(pixel);
            if(squared){
                pixel = pixel * pixel;
            }
            sum += pixel;
            int previ = i-1;
            int prevj = j-1;
            if(previ >= 0){
                sum += rtn[previ][j];
            }
            if(prevj >= 0){
                sum += column[prevj];
            }
            if(previ >= 0 && prevj >= 0){
                sum -= rtn[previ][prevj];
            }
            column.append(sum);
        }
        rtn.append(column);
    }
    return rtn;
}

QImage AdaptiveThresholder::AdaptivelyThreshold(int w, float bias){
    w = max(w, 1);
    qDebug() << w << '\n';
    QImage thresolded(srcImg.width(), srcImg.height(), QImage::Format_Mono);
    for(int i = 0; i < srcImg.width(); i++){
        for(int j = 0; j < srcImg.height(); j++){
            unsigned int pixel = srcImg.pixel(i, j);
            pixel = qGray(pixel) * bias;
            int thresholdValue = getThresholdValue(i, j, w);
            if(pixel >= thresholdValue){
                // White pixel
                thresolded.setPixel(i,j,1);
            }else{
                // Black pixel
                thresolded.setPixel(i,j,0);
            }
        }
    }
    return thresolded;
}

// Possibility to optimize here. Don't re-calculate mean here as well as in computing the variance...
int AdaptiveThresholder::getThresholdValue(int x, int y, int w, float k){
quint64 mean = getMean(x, y, w);
quint64 variance = getVariance(x, y, w);
quint64 std_dev = sqrt(variance);
int rtn = mean * (1 + k*(std_dev/128.0 -1));
return rtn;
}

// The width and height of the image MUST be greater than w. Otherwise undefined behavior (crash).
// If not enough surrounding region around the targeted pixel, will calculate the window mean with x,y as close to the
// center of the window as possible (though not exactly)
quint64 AdaptiveThresholder::getMean(int x, int y, int w){
    x = max(x, w/2);
    x = min(x, srcImg.width() - w/2 - 1);
    y = max(y, w/2);
    y = min(y, srcImg.height() - w/2 - 1);

    quint64 bottomRight = integralImg[x + w/2][y + w/2];
    quint64 topLeft = integralImg[x - w/2][y - w/2];
    quint64 topRight = integralImg[x + w/2][y - w/2];
    quint64 bottomLeft = integralImg[x - w/2][y + w/2];
    return (bottomRight + topLeft - topRight - bottomLeft) / (w*w);
}

// The width and height of the image MUST be greater than w. Otherwise undefined behavior (crash).
// If not enough surrounding region around the targeted pixel, will calculate the window mean with x,y as close to the
// center of the window as possible (though not exactly)
quint64 AdaptiveThresholder::getVariance(int x, int y, int w){
    x = max(x, w/2);
    x = min(x, srcImg.width() - w/2 - 1);
    y = max(y, w/2);
    y = min(y, srcImg.height() - w/2 - 1);

    quint64 bottomRight = squaredIntegralImg[x + w/2][y + w/2];
    quint64 topLeft = squaredIntegralImg[x - w/2][y - w/2];
    quint64 topRight = squaredIntegralImg[x + w/2][y - w/2];
    quint64 bottomLeft = squaredIntegralImg[x - w/2][y + w/2];

    quint64 firstTerm = (bottomRight + topLeft - topRight - bottomLeft) / (w*w);
    quint64 mean = getMean(x,y,w);
    return firstTerm - mean*mean;
}
